c $Id$
c=======================================================================
!---!   parameter and variable initializations
!---!
!---! authors Elizabeth C. Hunke, LANL
!---!         C. M. Bitz
c=======================================================================

      module ice_init

      use ice_domain

      implicit none

c=======================================================================

      contains

c=======================================================================

      subroutine input_data

!---!-------------------------------------------------------------------
!---! Namelist variables, set to default values; may be altered
!---! at run time
!---!
!---! author Elizabeth C. Hunke, LANL
!---!-------------------------------------------------------------------

      use ice_mechred
      use ice_diagnostics
      use ice_history
      use ice_calendar
      use ice_dyn_evp
      use ice_itd
      use ice_transport

      integer (kind=int_kind) :: nml_error ! namelist i/o error flag

      namelist /ice_nml/ year, istep0, dt, ndte, npt, 
     & diagfreq, histfreq, dumpfreq, restart, print_points,
     & kcolumn, kitd, kdyn, kstrength, evp_damping, snow_into_ocn,
     & grid_file, kmt_file, dump_file, restrt_file, history_file,
     & grid_type, advection, hist_avg

      !-----------------------------------------------------------------
      ! default values
      !-----------------------------------------------------------------
      year = 0001       ! initial year
      istep0 = 0        ! number of steps taken in previous integrations,
                        ! real (dumped) or imagined (use to set calendar)
      dt = 3600.0_dbl_kind  ! time step, s 
      ndte = 120        ! number of subcycles:  ndte=dt/dte
      npt = 99999       ! total number of time steps (dt) 
      diagfreq = 0      ! how often diag output is written
      histfreq='m'      ! output frequency
      dumpfreq='y'      ! restart frequency
      hist_avg = .true. ! write snapshots rather than time-averages
      print_points = .false.     ! if true, print point data
      kcolumn = 0       ! 1 = column model
      kitd = 1          ! type of itd conversions (0 = delta f, 1 = linear)
      kdyn = 1          ! type of dynamics (1 = evp)
      restart = .false. ! do not read restart files for initialization
      kstrength = 1     ! 1 = use Rothrock 1975 pressure formulation
      evp_damping = .false. ! if true, use damping procedure in evp dynamics
      snow_into_ocn = .false. ! if true, throw snow into ocn during ridging
      advection  = 'mpdata2'  ! second order advection using mpdata
      grid_type    = 'displaced_pole'  ! define rectangular grid internally

      grid_file    = 'data.domain.grid'
      kmt_file     = 'data.domain.kmt'
      dump_file    = 'iced'
      restrt_file  = 'ice.restart_file'
      history_file = 'iceh'

      !-----------------------------------------------------------------
      ! read from input file
      !-----------------------------------------------------------------
        open (21, file='ice_in', status='old')
   10   continue  !*** keep reading until right namelist is found
        read(21, nml=ice_nml, iostat=nml_error)
        if (nml_error > 0) goto 10    ! An error occurred
        if (nml_error < 0) goto 20    ! End of file condition
        close(21)
   20   continue
        if (nml_error /= 0) then
          if (my_task == master_task) then
            write (6,*) 'ice: Namelist read error in ice_init.F'
          endif
          stop
        endif

        if (histfreq.eq.'1') hist_avg = .false. ! potential conflict
        if (kcolumn.eq.1) grid_type    = 'column' 

      !-----------------------------------------------------------------
      ! spew
      !-----------------------------------------------------------------
      if (my_task.eq.master_task) then
        write (6,900) dt, ndte, npt, year, istep0
        write (6,901) diagfreq, histfreq, dumpfreq
        write (6,902) kitd, kdyn, kstrength, kcolumn
        write (6,*) ' '
        if (evp_damping) then
          write (6,*) 'Damping procedure used in evp dynamics'
        else
          write (6,*) 'Damping procedure NOT used in evp dynamics'
        endif
        write (6,*) ' '
        if (snow_into_ocn) then
          write (6,*) 'Snow will be thrown into ocn during ridging'
        else
          write (6,*) 'Snow will NOT be thrown into ocn during ridging'
        endif
        write (6,*) ' '
#ifdef coupled
#ifdef oceanmixed
        write (6,*) 'WARNING:  coupled and oceanmixed flags are BOTH ON'
        write (6,*) 'Ocean data received from coupler will be altered '
        write (6,*) 'by mixed layer routine! WARNING WARNING WARNING '
        write (6,*) ' '
#endif
#endif
        write (6,*) 'Advection algorithm is ',advection
        write (6,*) ' '
        write (6,*) 'Diagnostic data for the grid points specified'
        if (print_points) then
          write (6,*) 'in ice_diagnostics.H will be printed'
        else
          write (6,*) 'in ice_diagnostics.H will NOT be printed'
        endif
        write (6,*) ' '
        if (grid_type .eq. 'displaced_pole') then
          write (6,*) 'Displaced pole grid'
          write (6,*) 'The grid file is ',grid_file
          write (6,*) 'The land mask file is ', kmt_file
        elseif (grid_type .eq. 'column') then
          write (6,*) 'Column model grid'
        else
          write (6,*) 'Rectangular grid'
        endif

        write (6,*) 'The dump file prefix is ', dump_file
        if (restart) then
          write (6,*) 'The restart file pointer is ', restrt_file
        else
          write (6,*) 'Starting from default initialization '
          write (6,*) 'rather than restart file'
        endif
        write (6,*) 'The history file prefix is ', history_file
        if (hist_avg) then
          write (6,*) 'History data will be averaged over 1 ',histfreq
        else
          write (6,*) 'History data will be snapshots'
        endif
      endif

 900  format ('  dt=',f7.1,'  ndte=',i3,'  npt=',i6,
     &    '  year=',i8,'  istep0=',i8)
 901  format ('  diagfreq=',i3,'  histfreq=',a2,'  dumpfreq=',a2)
 902  format ('  kitd=',i2,'  kdyn=',i2,'  kstrength=',i2,
     &    '  kcolumn=',i2)

      end subroutine input_data

c=======================================================================

      subroutine init_state

!---!-------------------------------------------------------------------
!---! Initialize state for the itd model
!---!
!---! author C. M. Bitz
!---!-------------------------------------------------------------------

      use ice_model_size
      use ice_constants
      use ice_flux
      use ice_vthermo
      use ice_grid
      use ice_state
      use ice_itd
      use ice_dh

      integer (kind=int_kind) :: layer,nc,i,j,k
      real (kind=dbl_kind) ::
     &   slope, Ti
     &,  ainit(ncat)

      do nc=1,ncat
        do j=1,jmt_local
        do i=1,imt_local
          aicen(i,j,nc) = c0
          vicen(i,j,nc) = c0
          vsnon(i,j,nc) = c0
        enddo
        enddo
      enddo

      do nc=1,ncat
        Tsfcn(ilo:ihi,jlo:jhi,nc) = Tf(ilo:ihi,jlo:jhi)
        call bound(tsfcn(:,:,nc))
      enddo

      do layer=1,ntilay
        do j=1,jmt_local
        do i=1,imt_local
            eicen(i,j,layer) = c0
        enddo
        enddo
      enddo

#ifdef column
c initial ice conc and thick from sheba for end of Sept
      if (ncat.eq.1) then
      ainit(1)=.9008_dbl_kind
      else
      ainit(1)=0.2236_dbl_kind
      ainit(2)=0.3228_dbl_kind
      ainit(3)=0.2144_dbl_kind
      ainit(4)=0.1263_dbl_kind
      ainit(5)=0.0321_dbl_kind
      endif

      do j = jlo,jhi
      do i = ilo,ihi
        if (tmask(i,j)) then
        ! place ice where ocean sfc is cold
        if ( (sst (i,j).le.Tf(i,j)+p2) .and.
     &       (ULAT(i,j).lt.-40.0_dbl_kind/rad_to_deg .or.
     &        ULAT(i,j).gt. 40.0_dbl_kind/rad_to_deg)    )    then

          do nc = 1,ncat
            aicen(i,j,nc) = ainit(nc)
            if (nc.eq.1) vicen(i,j,nc) = 0.4403_dbl_kind*aicen(i,j,nc)   ! m
            if (nc.eq.2) vicen(i,j,nc) = 1.1167_dbl_kind*aicen(i,j,nc)   ! m
            if (nc.eq.3) vicen(i,j,nc) = 1.9726_dbl_kind*aicen(i,j,nc)   ! m
            if (nc.eq.4) vicen(i,j,nc) = 3.3278_dbl_kind*aicen(i,j,nc)   ! m
            if (nc.eq.5) vicen(i,j,nc) = 6.1351_dbl_kind*aicen(i,j,nc)   ! m
            vsnon(i,j,nc) = 0.05_dbl_kind*aicen(i,j,nc)   ! m
            
            Tsfcn(i,j,nc) = min(Tsmelt,Tair(i,j) - Tffresh)    ! deg C       
            ! make linear temp profile and compute enthalpy
            slope = Tf(i,j) - Tsfcn(i,j,nc)
            k=1
            do layer=layer1(nc),layern(nc)
              Ti=Tsfcn(i,j,nc)+slope*(real(k)-p5)/nilay(nc)
         eicen(i,j,layer)=energ(Ti,saltz(k,nc))*vicen(i,j,nc)/nilay(nc)
              k=k+1
            enddo
          enddo
        endif
        endif
      enddo
      enddo
#else
      if (ncat.eq.1) then
      ainit(1)=.8_dbl_kind
      else
      ainit(1)=.2_dbl_kind
      ainit(2)=.3_dbl_kind
      ainit(3)=.3_dbl_kind
      do nc=4,ncat
         ainit(nc)=0.1_dbl_kind
      enddo
      endif

      do j = jlo,jhi
      do i = ilo,ihi
        if (tmask(i,j)) then
        ! place ice where ocean sfc is cold
        if ( (sst (i,j).le.Tf(i,j)+p2) .and.
     &       (ULAT(i,j).lt.-40.0_dbl_kind/rad_to_deg .or.
     &        ULAT(i,j).gt. 40.0_dbl_kind/rad_to_deg)    )    then

          do nc = 1,ncat
            aicen(i,j,nc) = ainit(nc)
            if (nc.lt.ncat) then
              vicen(i,j,nc) = p5*(hin_max(nc-1)
     $           + hin_max(nc))*aicen(i,j,nc)   ! m
            else ! nc=ncat
              vicen(i,j,nc) = (hin_max(nc-1)+0.9_dbl_kind)*aicen(i,j,nc) ! m
            endif
            vsnon(i,j,nc) = p2*aicen(i,j,nc)   ! m
            
            Tsfcn(i,j,nc) = min(Tsmelt,Tair(i,j) - Tffresh)    ! deg C       
            ! make linear temp profile and compute enthalpy
            slope = Tf(i,j) - Tsfcn(i,j,nc)
            k=1
            do layer=layer1(nc),layern(nc)
              Ti=Tsfcn(i,j,nc)+slope*(real(k)-p5)/nilay(nc)
         eicen(i,j,layer)=energ(Ti,saltz(k,nc))*vicen(i,j,nc)/nilay(nc)
              k=k+1
            enddo
          enddo
        endif
        endif
      enddo
      enddo
#endif

      ! compute aggregate ice state and open water area
      call aggregate 

      end subroutine init_state

c=======================================================================

      subroutine set_state

!---!-------------------------------------------------------------------
!---! Prescribe ice state using input total ice concentration
!---!
!---! B. P. Briegleb  (12 Feb 2001)
!---!-------------------------------------------------------------------

      use ice_calendar
      use ice_model_size
      use ice_constants
      use ice_flux
      use ice_vthermo
      use ice_grid
      use ice_state
      use ice_itd
      use ice_dh

      integer (kind=int_kind) :: layer,nc,i,j,k
      real (kind=dbl_kind) ::
     &   slope, Ti, hi

      integer (kind=int_kind) ::
     &   m1,m2      ! two months    used in time interpolation
      real (kind=dbl_kind) :: 
     &   f1,f2      ! two fractions used in time interpolation
     &,  snwc(13)   ! mean snow cover (m) first of month (nrthrn hmsphr)
     &,  snwcnh     ! snow cover (m) for northern hemisphere
     &,  snwcsh     ! snow cover (m) for southern hemisphere
     &,  snw        ! actual snow cover (m) applied
      data snwc   /  .23,  .25,  .27,  .29,  .33,  .18,
     &                0.,   0.,  .02,  .12,  .18,  .21,  .23/ 

      ! set time parameters
      m1   = month
      m2   = month + 1
      f2   = (yday-daycal(m1))/(daycal(m2)-daycal(m1))
      f1   = c1 - f2

      ! compute hemispheric snow covers
      snwcnh = f1*snwc(m1) + f2*snwc(m2)
      m1 = m1 + 6
      if( m1 .gt. 13 ) m1 = m1 - 12
      m2 = m2 + 6
      if( m2 .gt. 13 ) m2 = m2 - 12
      snwcsh = f1*snwc(m1) + f2*snwc(m2)

      ! initialize ice state
      do nc=1,ncat
        do j=1,jmt_local
        do i=1,imt_local
          aicen(i,j,nc) = c0
          vicen(i,j,nc) = c0
          vsnon(i,j,nc) = c0
        enddo
        enddo
      enddo

      do nc=1,ncat
        Tsfcn(ilo:ihi,jlo:jhi,nc) = Tf(ilo:ihi,jlo:jhi)
        call bound(tsfcn(:,:,nc))
      enddo

      do layer=1,ntilay
        do j=1,jmt_local
        do i=1,imt_local
            eicen(i,j,layer) = c0
        enddo
        enddo
      enddo

      do j = jlo,jhi
      do i = ilo,ihi
        ! only over ocean
        if (tmask(i,j)) then
        ! place ice where ocean sfc is cold
        if ( (sst (i,j).le.Tf(i,j)+p2) ) then

          hi = 0.0_dbl_kind
          ! ice thickness 1.5m in Arctic and 0.5m in Antarctic
          if( ULAT(i,j).gt. 40.0_dbl_kind/rad_to_deg ) then
            hi  = 1.5_dbl_kind
            snw = snwcnh
          else if( ULAT(i,j).lt.-40.0_dbl_kind/rad_to_deg ) then
            hi  = 0.5_dbl_kind
            snw = snwcsh
          endif

          do nc = 1,ncat
            if( hin_max(nc-1).lt.hi .and. hi.lt.hin_max(nc) ) then
              aicen(i,j,nc) = 0.98_dbl_kind
              vicen(i,j,nc) = hi*aicen(i,j,nc) 
              ! keep snow/ice boundary above sea level by reducing snow
              vsnon(i,j,nc) = min(snw*aicen(i,j,nc),p33*vicen(i,j,nc))
              Tsfcn(i,j,nc) = min(Tair(i,j)-Tffresh,-p2)   ! deg C       
              ! make linear temp profile and compute enthalpy
              slope = Tf(i,j) - Tsfcn(i,j,nc)
              k=1
              do layer=layer1(nc),layern(nc)
                Ti=Tsfcn(i,j,nc)+slope*(real(k)-p5)/nilay(nc)
         eicen(i,j,layer)=energ(Ti,saltz(k,nc))*vicen(i,j,nc)/nilay(nc)
                k=k+1
              enddo
            endif
          enddo

        endif
        endif
      enddo
      enddo

      ! compute aggregate ice state and open water area
      call aggregate 

      end subroutine set_state

c=======================================================================

      subroutine init_flux

!---!-------------------------------------------------------------------
!---! initialize all fluxes exchanged with flux coupler
!---! and some data derived fields
!---!
!---! author Elizabeth C. Hunke, LANL
!---!-------------------------------------------------------------------

      use ice_constants
      use ice_flux

      integer i,j

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
      do i=ilo,ihi
      !-----------------------------------------------------------------
      ! fluxes received
      !-----------------------------------------------------------------
        zlvl(i,j) = c10    ! atm level height (m)
        uatm(i,j) = c5     ! wind velocity    (m/s)
        vatm(i,j) = c5
        potT(i,j) = 260.0_dbl_kind   ! air potential temperature  (K)
        rhoa(i,j) = 1.3_dbl_kind    ! air density (kg/m^3)
        Fsnow(i,j) = 3.3e-6_dbl_kind ! snowfall rate (kg/m2/s)
        Frain(i,j) = c0     ! rainfall rate (kg/m2/s)
        Fsw(i,j) = c0      ! shortwave radiation (W/m^2)
        Tair(i,j) = 260.0_dbl_kind   ! air temperature  (K)
        Qa(i,j) = 0.0014_dbl_kind   ! specific humidity (kg/kg)
        Flw(i,j) = 140.0_dbl_kind    ! incoming longwave radiation (W/m^2)
        sst(i,j) = -1.9_dbl_kind    ! sea surface temperature (C) 
        sss(i,j) = 34.0_dbl_kind     ! sea surface salinity (o/oo)
        uocn(i,j) = c0    ! surface ocean currents (m/s)
        vocn(i,j) = c0
        frzmlt(i,j)=-c2    ! freezing/melting potential (W/m^2)

      !-----------------------------------------------------------------
      ! derived or computed fields
      !-----------------------------------------------------------------

        Tf(i,j) = -depressT*sss(i,j)                      ! freezing temp (C)
        wind (i,j) = sqrt(uatm(i,j)**2 + vatm(i,j)**2) ! wind speed, (m/s)

        strocnx(i,j) = c0        ! ice-ocean stress, x-direction (U-cell)
        strocny(i,j) = c0        ! ice-ocean stress, y-direction (U-cell)
        strocnxT(i,j) = c0        ! ice-ocean stress, x-direction (T-cell)
        strocnyT(i,j) = c0        ! ice-ocean stress, y-direction (T-cell)

        Fhocn(i,j) = c0  

      call init_flux_atm(i,j)
      call init_flux_ocn(i,j)

      enddo
      enddo

      end subroutine init_flux

c=======================================================================

      subroutine set_flux

!---!-------------------------------------------------------------------
!---! set fluxes for dummy ice model
!---!
!---! author Bruce P. Briegleb, NCAR
!---!-------------------------------------------------------------------

      use ice_constants
      use ice_dyn_evp
      use ice_flux

      integer i,j

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
      do i=ilo,ihi
      !-----------------------------------------------------------------
      ! fluxes received
      !-----------------------------------------------------------------
        Fsnow(i,j)    = c0     ! snowfall rate (kg/m2/s)
        Frain(i,j)    = c0     ! rainfall rate (kg/m2/s)
        frzmlt(i,j)   = c0     ! freezing/melting potential (W/m^2)
      !-----------------------------------------------------------------
      ! ice velocities
      !-----------------------------------------------------------------
        u(i,j)        = c0     ! ice velocity u
        v(i,j)        = c0     ! ice velocity v
      !-----------------------------------------------------------------
      ! derived or computed fields
      !-----------------------------------------------------------------
        strocnxT(i,j) = c0     ! ice-ocean stress, x-direction (T-cell)
        strocnyT(i,j) = c0     ! ice-ocean stress, y-direction (T-cell)
      !-----------------------------------------------------------------
      ! other atm and ocn fluxes
      !-----------------------------------------------------------------
        call init_flux_atm(i,j)
        call init_flux_ocn(i,j)
      enddo
      enddo

      end subroutine set_flux

c=======================================================================

      subroutine setup_mpi

!---!-------------------------------------------------------------------
!---! this routine initializes mpi for either internal parallel
!---! processing or for message passing with the coupler
!---!
!---! author Elizabeth C. Hunke, LANL
!---! code originally based on POP routine
!---!-------------------------------------------------------------------

      use ice_mpi_internal
      use ice_coupling

      integer (kind=int_kind) ::
     &  ice_task           ! master task for ice
     &, coords1, coords2, n, ilen, jlen
     &, interior_i, interior_j  ! dummies for interior blocks

      master_task = 0

#ifdef coupled
      !  if running in coupled mode
      call MPI_INIT(ierr)
      call mpi_coupled ('ice', cpl_task, ice_task, MPI_COMM_ICE)
#else
      !  if running in stand-alone MPI mode
#ifdef _MPI
#if fcd_coupled
#else
      call MPI_INIT(ierr)
#endif
      call MPI_COMM_DUP(MPI_COMM_WORLD, MPI_COMM_ICE, ierr)
#endif
#endif

#ifdef _MPI
      call MPI_COMM_SIZE (MPI_COMM_ICE, nb_tasks, ierr)
      call MPI_COMM_RANK (MPI_COMM_ICE, my_task, ierr)
      if (nb_tasks.ne.nproc_s) 
     &  write (6,*) ' error in nb_tasks ',nb_tasks,nproc_s
      if ( ierr /= MPI_SUCCESS ) then 
         write (6,*) '(setup_mpi) ERROR after MPI_COMM_xxx'
         call MPI_FINALIZE(ierr)
         stop
      endif

      coords1=mod(my_task,nproc_x)
      coords2=my_task/nproc_x
      nbr_east = coords2*nproc_x+mod(my_task+1,nproc_x)
      nbr_west = coords2*nproc_x+mod(my_task-1+nproc_x,nproc_x)
      nbr_north = my_task+nproc_x
      nbr_south = my_task-nproc_x
      if (nbr_south.lt.0) nbr_south = -1
      if (nbr_north.gt.nproc_s-1) nbr_north=-1

      ilen=ihi-ilo+1
      jlen=jhi-jlo+1

      do n=1,nproc_s

      local_start(1,n)=((imt_global-1)/nproc_x+1)*mod((n-1),nproc_x)+1
      local_start(2,n)=((jmt_global-1)/nproc_y+1)*((n-1)/nproc_x)+1

      call MPI_TYPE_VECTOR(jlen, ilen, ilen, 
     &     mpi_integer, mpi_interior_int(n), ierr)
      call MPI_TYPE_COMMIT(mpi_interior_int(n), ierr)

      call MPI_TYPE_VECTOR(jlen, ilen, ilen, 
     &     mpi_real8, mpi_interior_real(n), ierr)
      call MPI_TYPE_COMMIT(mpi_interior_real(n), ierr)

      call MPI_TYPE_VECTOR(jlen, ilen, imt_global, 
     &     mpi_integer, mpi_interior_int_global(n), ierr)
      call MPI_TYPE_COMMIT(mpi_interior_int_global(n), ierr)

      call MPI_TYPE_VECTOR(jlen, ilen, imt_global, 
     &     mpi_real8, mpi_interior_real_global(n), ierr)
      call MPI_TYPE_COMMIT(mpi_interior_real_global(n), ierr)

      enddo

      do n=1,nproc_s
      if (my_task.eq.n-1) then
      write (6,*) ' my_task,e,w,n,s ',my_task,nbr_east,nbr_west,
     &  nbr_north,nbr_south
      endif
      enddo
#else
      my_task = master_task
      nb_tasks = 1 
#endif

      end subroutine setup_mpi

c=======================================================================

      end module ice_init

c=======================================================================

