c $Id$
c=======================================================================
!---! spatial grids, masks, and boundary conditions
!---!
!---! authors Elizabeth C. Hunke, LANL
!---!         Tony Craig, NCAR
c=======================================================================

      module ice_grid

      use ice_kinds_mod
      use ice_constants
      use ice_domain

      implicit none

      character (len=char_len) :: 
     &   grid_file         !  input file for POP grid info
     &,  kmt_file          !  input file for POP grid info
     &,  grid_type         !  rectangular (default) or displaced_pole 

      real (kind=dbl_kind), dimension (imt_local,jmt_local) ::
     &   dxt  ! width of T-cell through the middle (m)
     &,  dyt  ! height of T-cell through the middle (m)
     &,  dxu  ! width of U-cell through the middle (m)
     &,  dyu  ! height of U-cell through the middle (m)
     &,  HTE  ! length of eastern edge of T-cell (m)
     &,  HTN  ! length of northern edge of T-cell (m)
     &,  tarea! area of T-cell (m^2)
     &,  uarea! area of U-cell (m^2)
     &,  ULON ! longitude of velocity pts (radians)
     &,  ULAT ! latitude of velocity pts (radians)

      real (kind=dbl_kind), dimension (ilo:ihi,jlo:jhi) ::
     &   dxtr     ! 0.5/dxt
     &,  dxtr4    ! 0.25/dxt
     &,  dytr     ! 0.5/dyt
     &,  dytr4    ! 0.25/dyt
     &,  ANGLE    ! for conversions between POP grid and lat/lon
     &,  ANGLET   ! ANGLE converted to T-cells
     &,  TLON       ! t-grid center longitude
     &,  TLAT       ! t-grid center latitude
     &,  tarean      ! area of NH cells
     &,  tareas      ! area of SH cells

      ! Masks
      real (kind=dbl_kind), dimension (imt_local,jmt_local) ::
     &   hm   ! land/boundary mask, thickness (T-cell)
     &,  uvm  ! land/boundary mask, velocity (U-cell)
     &,  mask_n   ! northern hemisphere
     &,  mask_s   ! southern hemisphere

      logical (kind=log_kind) ::
     &   tmask(imt_local,jmt_local) ! land/boundary mask, thickness (T-cell)
     &,  umask(imt_local,jmt_local) ! land/boundary mask, velocity (U-cell)
     &,  icetmask(ilo:ihi,jlo:jhi)  ! ice extent mask (T-cell)
     &,  iceumask(ilo:ihi,jlo:jhi)  ! ice extent mask (U-cell)

c=======================================================================

      contains

c=======================================================================

      subroutine init_grid

!---!-------------------------------------------------------------------
!---! Horizontal grid initialization
!---!     HT{N,E} = cell widths on {N,E} sides of T cell
!---!     U{LAT,LONG} = true {latitude,longitude} of U points
!---!     D{X,Y}{T,U} = {x,y} spacing centered at {T,U} points
!---!
!---! author Elizabeth C. Hunke, LANL
!---!-------------------------------------------------------------------

      integer (kind=int_kind) :: i, j
      real (kind=dbl_kind) ::
     &     tmpg(imt_global,jmt_global)   ! temporary array
     &,    tmpg4(4,imt_global,jmt_global)   ! temporary array

      if (grid_type .eq. 'displaced_pole') then
        call popgrid              ! read POP grid lengths directly
      elseif (grid_type .eq. 'column') then
        call columngrid           ! column model grid
      else
        call rectgrid             ! regular rectangular grid
      endif

      call bound(HTN)
      call bound(HTE)
      call bound(ULAT)
      call bound(ULON)

      !-----------------------------------------------------------------
      ! construct T-grid cell and U-grid cell widths
      !-----------------------------------------------------------------

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
        dxt(i,j) = p5*(HTN(i,j) + HTN(i,j-1))
        dyt(i,j) = p5*(HTE(i,j) + HTE(i-1,j))

        dxtr(i,j) = c1/dxt(i,j)
        dytr(i,j) = c1/dyt(i,j)
        dxtr4(i,j) = p25/dxt(i,j)
        dytr4(i,j) = p25/dyt(i,j)

        tarea(i,j) = dxt(i,j)*dyt(i,j)

        dxu(i,j) = p5*(HTN(i,j) + HTN(i+1,j))
        dyu(i,j) = p5*(HTE(i,j) + HTE(i,j+1))
       enddo
      enddo

      call bound(dxt)
      call bound(dyt)
      call bound(dxu)
      call bound(dyu)
      call bound(tarea)

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
        uarea(i,j) = p25*(tarea(i,j) + tarea(i+1,j)
     &         + tarea(i,j+1) + tarea(i+1,j+1))        ! m^2
       enddo
      enddo
      call bound(uarea)

      call u2tgrid(ANGLET)   ! ANGLE on the T grid

#ifndef coupled
       call Tlatlon(tmpg,tmpg,tmpg4,tmpg4) ! TLAT, TLON
#endif

      call makemask              ! velocity mask, hemisphere masks

      end subroutine init_grid

c=======================================================================

      subroutine popgrid

!---!-------------------------------------------------------------------
!---! POP displaced pole grid and land mask
!---!   grid
!---!      rec no.         field         units
!---!      -------         -----         -----
!---!         1            ULAT         radians
!---!         2            ULON         radians
!---!         3             HTN           cm
!---!         4             HTE           cm
!---!         5             HUS           cm
!---!         6             HUW           cm
!---!         7            ANGLE        radians
!---!   land mask
!---!      rec no.         field         units
!---!      -------         -----         -----
!---!         1             KMT
!---!
!---! author Elizabeth C. Hunke, LANL
!---!-------------------------------------------------------------------

      use ice_read_write

      integer (kind=int_kind) :: i, j
      real (kind=dbl_kind) :: work(ilo:ihi,jlo:jhi)
      logical (kind=log_kind) :: scatter

      call ice_open(11,grid_file,64)
      call ice_open(12,kmt_file,32)

      scatter = .true.

      call ice_read(11,1,work,'rda8',scatter)
      ULAT(ilo:ihi,jlo:jhi)=work(ilo:ihi,jlo:jhi)
      call ice_read(11,2,work,'rda8',scatter)
      ULON(ilo:ihi,jlo:jhi)=work(ilo:ihi,jlo:jhi)
      call ice_read(11,3,work,'rda8',scatter)
      HTN(ilo:ihi,jlo:jhi)=work(ilo:ihi,jlo:jhi)*cm_to_m
      call ice_read(11,4,work,'rda8',scatter)
      HTE(ilo:ihi,jlo:jhi)=work(ilo:ihi,jlo:jhi)*cm_to_m
      call ice_read(11,7,work,'rda8',scatter)
      ANGLE(ilo:ihi,jlo:jhi)=work(ilo:ihi,jlo:jhi)
      call ice_read(12,1,work,'ida4',scatter)

      if (my_task.eq.master_task) then
         close (11) 
         close (12) 
      endif

      do j=jlo,jhi
       do i=ilo,ihi
         hm(i,j) = work(i,j)
         if (hm(i,j).ge.c1) hm(i,j) = c1 
         ANGLET(i,j) = ANGLE(i,j)   
       enddo
      enddo

      end subroutine popgrid

c=======================================================================

      subroutine columngrid

!---!-------------------------------------------------------------------
!---! Column grid and mask
!---!
!---! author C. M. Bitz (based on rectgrid by Hunke)
!---!-------------------------------------------------------------------

      use ice_model_size
      use ice_mpi_internal

      integer (kind=int_kind) :: i, j
      real (kind=dbl_kind) :: 
     &   hmg(imt_global,jmt_global)
     &,  work(ilo:ihi,jlo:jhi)

c.. calculate various geometric 2d arrays
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
         HTN  (i,j) = c1           ! constant longitude spacing = 
                                   ! meaningless
         HTE  (i,j) = c1           ! constant latitude  spacing = 
                                   ! meaningless
         ULAT (i,j) = 75.5/rad_to_deg    ! used to determine hemisphere and 
         ULON (i,j) = 170.0/rad_to_deg   ! init_state, need not be exact
         ANGLE(i,j) = c0      ! "square with the world"
         ANGLET(i,j) = c0 
       enddo
      enddo


c.. verify that imt_global and jmt_global are 1
      if ((imt_global.ne.1).or. (jmt_global.ne.1)) then
         write(6,*) 'Because you have selected the column model flag'
         write(6,*) 'Please set imt_global=jmt_global=1 in file'
         write(6,*) 'ice_model_size.F and recompile'
         stop
      endif

      !-----------------------------------------------------------------
      ! construct T-cell land mask
      !-----------------------------------------------------------------
c$OMP PARALLEL DO PRIVATE(i,j)
        do j=1,jmt_global        ! initialize hm
         do i=1,imt_global       ! open
          hmg(i,j) = c0
         enddo
        enddo

c$OMP PARALLEL DO PRIVATE(i,j)
        do j=1,jmt_global        ! open
         do i=1,imt_global       ! open
          hmg(i,j) = c1
         enddo
        enddo

      call global_scatter(hmg,work)

c$OMP PARALLEL DO PRIVATE(i,j)
        do j=jlo,jhi
         do i=ilo,ihi
          hm(i,j) = work(i,j)
         enddo
        enddo

      end subroutine columngrid

c=======================================================================

      subroutine rectgrid

!---!-------------------------------------------------------------------
!---! Regular rectangular grid and mask
!---!
!---! author Elizabeth C. Hunke, LANL
!---!-------------------------------------------------------------------

      use ice_model_size
      use ice_mpi_internal

      integer (kind=int_kind) :: i, j
      real (kind=dbl_kind) :: 
     &   hmg(imt_global,jmt_global)
     &,  work(ilo:ihi,jlo:jhi)

c.. calculate various geometric 2d arrays
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
c         HTN  (i,j) = 3.1e4_dbl_kind  ! constant longitude spacing =  
                                       ! POP <4/3> min, m
c         HTE  (i,j) = 3.1e4_dbl_kind  ! constant latitude  spacing = 
                                       ! POP <4/3> min, m
         HTN  (i,j) = 1.6e4_dbl_kind   ! constant longitude spacing = 
                                       ! POP <2/3> min, m
         HTE  (i,j) = 1.6e4_dbl_kind   ! constant latitude  spacing = 
                                       ! POP <2/3> min, m
         ULAT (i,j) = c0      ! remember to set Coriolis !
         ULON (i,j) = c0
         ANGLE(i,j) = c0      ! "square with the world"
         ANGLET(i,j) = c0 
       enddo
      enddo

      !-----------------------------------------------------------------
      ! construct T-cell land mask
      !-----------------------------------------------------------------
c$OMP PARALLEL DO PRIVATE(i,j)
        do j=1,jmt_global        ! initialize hm
         do i=1,imt_global       ! open
          hmg(i,j) = c0
         enddo
        enddo

c        do j=1,jmt_global        ! open
c         do i=1,imt_global       ! open
c$OMP PARALLEL DO PRIVATE(i,j)
        do j=3,jmt_global-2       ! closed: NOTE jmt_global > 5
         do i=3,imt_global-2      ! closed: NOTE imt_global > 5
          hmg(i,j) = c1
         enddo
        enddo

      call global_scatter(hmg,work)

c$OMP PARALLEL DO PRIVATE(i,j)
        do j=jlo,jhi
         do i=ilo,ihi
          hm(i,j) = work(i,j)
         enddo
        enddo

      end subroutine rectgrid

c=======================================================================

      subroutine makemask

!---!-------------------------------------------------------------------
!---! Sets the boundary values for the T cell land mask (hm) and
!---! makes the logical land masks for T and U cells (tmask, umask).
!---! Also creates hemisphere masks (mask_n northern, mask_s southern)
!---!
!---! author Elizabeth C. Hunke, LANL
!---!-------------------------------------------------------------------

      integer (kind=int_kind) :: i, j

      call bound(hm)  !!! use real arrays to get boundary conditions

      !-----------------------------------------------------------------
      ! construct T-cell and U-cell masks
      !-----------------------------------------------------------------

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
        uvm(i,j) = min(hm(i,j),hm(i+1,j),hm(i,j+1),hm(i+1,j+1))
       enddo
      enddo
      call bound(uvm)  !!! use real arrays to get boundary conditions

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=1,jmt_local
       do i=1,imt_local
        tmask(i,j) = .false.
        umask(i,j) = .false.
        if ( hm(i,j).gt.p5) tmask(i,j) = .true. 
        if (uvm(i,j).gt.p5) umask(i,j) = .true. 
       enddo
      enddo

      !-----------------------------------------------------------------
      ! create hemisphere masks
      !-----------------------------------------------------------------

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=1,jmt_local
       do i=1,imt_local
        mask_n(i,j) = c0
        mask_s(i,j) = c0
       enddo
      enddo
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
        if (ULAT(i,j).ge.-puny) mask_n(i,j) = c1  ! northern hemisphere
        if (ULAT(i,j).lt.-puny) mask_s(i,j) = c1  ! southern hemisphere

        tarean(i,j) = tarea(i,j)*mask_n(i,j)  ! N hemisphere area mask (m^2)
        tareas(i,j) = tarea(i,j)*mask_s(i,j)  ! S hemisphere area mask (m^2)
       enddo
      enddo

      end subroutine makemask

c=======================================================================

      subroutine Tlatlon(TLAT_G,TLON_G,TLAT_V,TLON_V)

!---!-------------------------------------------------------------------
!---! initializes lat and lon on T grid
!---!
!---! author Elizabeth C. Hunke, LANL
!---! code originally based on POP routine
!---!-------------------------------------------------------------------
      use ice_model_size
      use ice_mpi_internal

      real (kind=dbl_kind), intent(out) ::
     &     TLAT_G(imt_global,jmt_global)   ! latitude  of cell center
     &,    TLON_G(imt_global,jmt_global)   ! longitude of cell center
     &,    TLAT_V(4,imt_global,jmt_global) ! latitude  of cell vertices
     &,    TLON_V(4,imt_global,jmt_global) ! longitude of cell vertices

      integer (kind=int_kind) ::
     &   i, j                          ! generic indices

      real (kind=dbl_kind) ::
     &     LSW(ilo:ihi,jlo:jhi)          ! longitude of SW corner
     &,    LSE(ilo:ihi,jlo:jhi)          ! longitude of SE corner
     &,    LNW(ilo:ihi,jlo:jhi)          ! longitude of NW corner
     &,    LNE(ilo:ihi,jlo:jhi)          ! longitude of NE corner

      !-----------------------------------------------------------------
      !     initialize grid center and vertex coordinates in degrees 
      !-----------------------------------------------------------------

      ! longitude

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
        do i=ilo,ihi
            LSE(i,j) = ULON(i  ,j-1)
            LNW(i,j) = ULON(i-1,j  )
            LSW(i,j) = ULON(i-1,j-1)

      ! for longitude, must be careful of longitude branch cut (eg 0,2pi)
          if (LSE(i,j) - ULON(i,j) .gt.  c4) LSE(i,j) = 
     &                                           LSE(i,j) - pi2
          if (LSE(i,j) - ULON(i,j) .lt. -c4) LSE(i,j) = 
     &                                           LSE(i,j) + pi2
          if (LNW(i,j) - ULON(i,j) .gt.  c4) LNW(i,j) = 
     &                                           LNW(i,j) - pi2
          if (LNW(i,j) - ULON(i,j) .lt. -c4) LNW(i,j) = 
     &                                           LNW(i,j) + pi2
          if (LSW(i,j) - ULON(i,j) .gt.  c4) LSW(i,j) = 
     &                                           LSW(i,j) - pi2
          if (LSW(i,j) - ULON(i,j) .lt. -c4) LSW(i,j) = 
     &                                           LSW(i,j) + pi2
 
          LNE(i,j) = rad_to_deg*ULON (i,j)
          LNW(i,j) = rad_to_deg*LNW(i,j)
          LSW(i,j) = rad_to_deg*LSW(i,j)
          LSE(i,j) = rad_to_deg*LSE(i,j)

          TLON(i,j) = p25*(LNE(i,j) + LNW(i,j)
     &                   + LSE(i,j) + LSW(i,j))
          if (TLON(i,j) .gt. c360) TLON(i,j) = TLON(i,j) - c360
          if (TLON(i,j) .lt. c0 )  TLON(i,j) = TLON(i,j) + c360

        end do
      end do

      call global_gather(TLON_G,LSW) ! use TLON_G as temp space

      if (my_task == master_task) then
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=1,jmt_global
        do i=1,imt_global
          TLON_V(1,i,j) = TLON_G(i,j)
        end do
      end do
      endif

      call global_gather(TLON_G,LSE)

      if (my_task == master_task) then
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=1,jmt_global
        do i=1,imt_global
          TLON_V(2,i,j) = TLON_G(i,j)
        end do
      end do
      endif

      call global_gather(TLON_G,LNE)

      if (my_task == master_task) then
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=1,jmt_global
        do i=1,imt_global
          TLON_V(3,i,j) = TLON_G(i,j)
        end do
      end do
      endif

      call global_gather(TLON_G,LNW)

      if (my_task == master_task) then
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=1,jmt_global
        do i=1,imt_global
          TLON_V(4,i,j) = TLON_G(i,j)
          TLON_G(i,j)   = p25*(TLON_V(1,i,j) + TLON_V(2,i,j) +
     &                          TLON_V(3,i,j) + TLON_V(4,i,j))
        end do
      end do
      endif
  
      ! latitude  (use LON* as work array)

      do j=jlo,jhi
        do i=ilo,ihi
          LNE(i,j) = rad_to_deg*ULAT(i  ,j  )
          LSE(i,j) = rad_to_deg*ULAT(i  ,j-1)
          LNW(i,j) = rad_to_deg*ULAT(i-1,j  )
          LSW(i,j) = rad_to_deg*ULAT(i-1,j-1)

          TLAT(i,j) = p25*(LNE(i,j) + LNW(i,j)
     &                   + LSE(i,j) + LSW(i,j))
        end do
      end do

      call global_gather(TLAT_G,LSW)  ! use TLAT_G as temp space

      if (my_task == master_task) then
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=1,jmt_global
        do i=1,imt_global
          TLAT_V(1,i,j) = TLAT_G(i,j)
        end do
      end do
      endif

      call global_gather(TLAT_G,LSE)

      if (my_task == master_task) then
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=1,jmt_global
        do i=1,imt_global
          TLAT_V(2,i,j) = TLAT_G(i,j)
        end do
      end do
      endif

      call global_gather(TLAT_G,LNE)

      if (my_task == master_task) then
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=1,jmt_global
        do i=1,imt_global
          TLAT_V(3,i,j) = TLAT_G(i,j)
        end do
      end do
      endif

      call global_gather(TLAT_G,LNW)

      if (my_task == master_task) then
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=1,jmt_global
        do i=1,imt_global
          TLAT_V(4,i,j) = TLAT_G(i,j)
          TLAT_G(i,j)   = p25*(TLAT_V(1,i,j) + TLAT_V(2,i,j) +
     &                         TLAT_V(3,i,j) + TLAT_V(4,i,j))
        end do
      end do
      endif ! master_task

      end subroutine Tlatlon

c=======================================================================

      subroutine t2ugrid(work)

!---!-------------------------------------------------------------------
!---! transfer from T-cell centers to U-cell centers
!---! writes work into another array that has ghost cells
!---!
!---! author Elizabeth C. Hunke, LANL
!---!-------------------------------------------------------------------

      integer (kind=int_kind) :: i, j
      real (kind=dbl_kind) :: work (ilo:ihi,jlo:jhi)
     &,   work1(imt_local,jmt_local)

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
        work1(i,j) = work(i,j)
       enddo
      enddo
      call bound(work1)
      call to_ugrid(work1,work)

      end subroutine t2ugrid

c=======================================================================

      subroutine to_ugrid(work1,work2)

!---!-------------------------------------------------------------------
!---! shifts quantities from the T-cell midpoint (work1) to the U-cell 
!---! midpoint (work2)
!---!
!---! author Elizabeth C. Hunke, LANL
!---!-------------------------------------------------------------------

      integer (kind=int_kind) :: i, j
      real (kind=dbl_kind) :: work1(imt_local,jmt_local)
     &,   work2(ilo:ihi,jlo:jhi)

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
       work2(i,j) = p25*(work1(i,j)*tarea(i,j) 
     &                   + work1(i+1,j)*tarea(i+1,j)
     &                   + work1(i,j+1)*tarea(i,j+1) 
     &                   + work1(i+1,j+1)*tarea(i+1,j+1))/uarea(i,j)
       enddo
      enddo

      end subroutine to_ugrid

c=======================================================================

      subroutine u2tgrid(work)

!---!-------------------------------------------------------------------
!---! transfer from U-cell centers to T-cell centers
!---! writes work into another array that has ghost cells
!---!
!---! author Elizabeth C. Hunke, LANL
!---!-------------------------------------------------------------------

      integer (kind=int_kind) :: i, j
      real (kind=dbl_kind) :: work (ilo:ihi,jlo:jhi)
     &,   work1(imt_local,jmt_local)

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
        work1(i,j) = work(i,j)
       enddo
      enddo
      call bound(work1)
      call to_tgrid(work1,work)

      end subroutine u2tgrid

c=======================================================================

      subroutine to_tgrid(work1,work2)

!---!-------------------------------------------------------------------
!---! shifts quantities from the U-cell midpoint (work1) to the T-cell 
!---! midpoint (work2)
!---!
!---! author Elizabeth C. Hunke, LANL
!---!-------------------------------------------------------------------

      integer (kind=int_kind) :: i, j
      real (kind=dbl_kind) :: work1(imt_local,jmt_local)
     &,   work2(ilo:ihi,jlo:jhi)

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
       work2(i,j) = p25*(work1(i,j)*uarea(i,j) 
     &                   + work1(i-1,j)*uarea(i-1,j)
     &                   + work1(i,j-1)*uarea(i,j-1) 
     &                   + work1(i-1,j-1)*uarea(i-1,j-1))/tarea(i,j)
       enddo
      enddo

      end subroutine to_tgrid

c=======================================================================

      subroutine bound(work1)

!---!-------------------------------------------------------------------
!---! fills ghost cells with boundary information
!---!
!---! author Tony Craig, NCAR
!---!-------------------------------------------------------------------

      real (kind=dbl_kind) :: work1(1)
      call bound_ijn(1,work1,.true.,.true.,.true.,.true.)

      end subroutine bound

c=======================================================================

      subroutine bound_sw(work1)

!---!-------------------------------------------------------------------
!---! fills south and west ghost cells with boundary information
!---!
!---! author Tony Craig, NCAR
!---!-------------------------------------------------------------------

      real (kind=dbl_kind) :: work1(1)
      call bound_ijn(1,work1,.false.,.true.,.false.,.true.)

      end subroutine bound_sw

c=======================================================================

      subroutine bound_narr_ne(narrays,work1)

!---!-------------------------------------------------------------------
!---! fills north and east ghost cells with boundary information
!---!
!---! NOTE: work1 array has form (number_arrays,i_index,j_index)
!---!       for evp dynamics performance
!---!
!---! authors Tony Craig, NCAR
!---!         Elizabeth Hunke, LANL
!---!-------------------------------------------------------------------

      integer (kind=int_kind) :: narrays
      real (kind=dbl_kind) :: work1(1)
      call bound_nij(narrays,work1,.true.,.false.,.true.,.false.)

      end subroutine bound_narr_ne

c=======================================================================

      subroutine bound_narr(narrays,work1)

!---!-------------------------------------------------------------------
!---! fills north and east ghost cells with boundary information
!---! narr arrays at once (for performance)
!---!
!---! authors Tony Craig, NCAR
!---!         Elizabeth Hunke, LANL
!---!-------------------------------------------------------------------

      integer (kind=int_kind) :: narrays
      real (kind=dbl_kind) :: work1(1)
      call bound_ijn(narrays,work1,.true.,.true.,.true.,.true.)

      end subroutine bound_narr

c=======================================================================

      subroutine bound_ijn(nd,work1,north,south,east,west)

!---!-------------------------------------------------------------------
!---! Periodic/Neumann conditions for global domain boundaries
!---! Assumptions:  a *single* row of ghost cells (num_ghost_cells=1)
!---!
!---! work1 array has form (i_index,j_index,number_arrays)
!---!
!---! authors Tony Craig, NCAR
!---!         Elizabeth Hunke, LANL
!---!-------------------------------------------------------------------

      use ice_timers
      use ice_mpi_internal

      integer (kind=int_kind) :: i, j, nd, n
      real (kind=dbl_kind) :: work1(imt_local,jmt_local,nd)
      logical north,south,east,west
#ifdef _MPI
      integer (kind=int_kind) :: icnt,jcnt
     &,   status(MPI_STATUS_SIZE),request(4)
      real (kind=dbl_kind) :: workw(jlo:jhi,nd),worke(jlo:jhi,nd)
     &,  workn(ilo-1:ihi+1,nd),works(ilo-1:ihi+1,nd)
#endif

      call ice_timer_start(10) ! bound

#ifdef _MPI
      jcnt=(jhi-jlo+1)*nd
      icnt=(ihi-ilo+1+2*num_ghost_cells)*nd

      !-----------------------------------------------------------------
      ! west data to east data, west shift
      !-----------------------------------------------------------------
      if (east) then

      do n=1,nd
      do j=jlo,jhi
      workw(j,n)=work1(ilo,j,n)
      enddo
      enddo

      call MPI_SENDRECV(workw,jcnt,MPI_REAL8,nbr_west,my_task,
     &                  worke,jcnt,MPI_REAL8,nbr_east,nbr_east,
     &                  MPI_COMM_ICE,status,ierr)

      do n=1,nd
      do j=jlo,jhi
      work1(ihi+1,j,n)=worke(j,n)
      enddo
      enddo

      endif

      !-----------------------------------------------------------------
      ! east data to west data, east shift
      !-----------------------------------------------------------------
      if (west) then

      do n=1,nd
      do j=jlo,jhi
      worke(j,n)=work1(ihi,j,n)
      enddo
      enddo

      call MPI_SENDRECV(worke,jcnt,MPI_REAL8,nbr_east,my_task,
     &                  workw,jcnt,MPI_REAL8,nbr_west,nbr_west,
     &                  MPI_COMM_ICE,status,ierr)

      do n=1,nd
      do j=jlo,jhi
      work1(ilo-1,j,n)=workw(j,n)
      enddo
      enddo

      endif

      !-----------------------------------------------------------------
      ! north data to south data, north shift
      !-----------------------------------------------------------------
      if (south) then
      if (nbr_south.ne.-1) then
        call MPI_IRECV(works,
     &                 icnt,MPI_REAL8,nbr_south,nbr_south,
     &                 MPI_COMM_ICE,request(1),ierr)
      else

        do n=1,nd
        do i=ilo-1,ihi+1
        work1(i,jlo-1,n)=work1(i,jlo,n)
        enddo
        enddo

      endif
      if (nbr_north.ne.-1) then

        do n=1,nd
        do i=ilo-1,ihi+1
        workn(i,n)=work1(i,jhi,n)
        enddo
        enddo

        call MPI_ISEND (workn,
     &                 icnt,MPI_REAL8,nbr_north,my_task,
     &                 MPI_COMM_ICE,request(2),ierr)
      endif
      if (nbr_north.ne.-1) then
        call MPI_WAIT(request(2), status, ierr)
      endif
      if (nbr_south.ne.-1) then
        call MPI_WAIT(request(1), status, ierr)

        do n=1,nd
        do i=ilo-1,ihi+1
        work1(i,jlo-1,n)=works(i,n)
        enddo
        enddo

      endif
      endif

      !-----------------------------------------------------------------
      ! south data to north data, south shift
      !-----------------------------------------------------------------
      if (north) then
      if (nbr_north.ne.-1) then
        call MPI_IRECV(workn,
     &                 icnt,MPI_REAL8,nbr_north,nbr_north,
     &                 MPI_COMM_ICE,request(3),ierr)
      else

        do n=1,nd
        do i=ilo-1,ihi+1
        work1(i,jhi+1,n)=work1(i,jhi,n)
        enddo
        enddo

      endif
      if (nbr_south.ne.-1) then

        do n=1,nd
        do i=ilo-1,ihi+1
        works(i,n)=work1(i,jlo,n)
        enddo
        enddo

        call MPI_ISEND (works,
     &                 icnt,MPI_REAL8,nbr_south,my_task,
     &                 MPI_COMM_ICE,request(4),ierr)
      endif
      if (nbr_south.ne.-1) then
        call MPI_WAIT(request(4), status, ierr)
      endif
      if (nbr_north.ne.-1) then
        call MPI_WAIT(request(3), status, ierr)

        do n=1,nd
        do i=ilo-1,ihi+1
        work1(i,jhi+1,n)=workn(i,n)
        enddo
        enddo

      endif
      endif

#else
      !-----------------------------------------------------------------
      ! single domain
      !-----------------------------------------------------------------
 
      do n=1,nd

      ! Periodic conditions
c$OMP PARALLEL DO PRIVATE(j)
      do j=jlo,jhi
       work1(ilo-1,j,n) = work1(ihi,j,n)
       work1(ihi+1,j,n) = work1(ilo,j,n)
      enddo

      ! Neumann conditions (POP grid land points)
c$OMP PARALLEL DO PRIVATE(i)
      do i=ilo-1,ihi+1
        work1(i,jlo-1,n) = work1(i,jlo,n)
        work1(i,jhi+1,n) = work1(i,jhi,n)
      enddo

      enddo  ! n
#endif
      call ice_timer_stop(10)  ! bound

      end subroutine bound_ijn

c=======================================================================

      subroutine bound_nij(nd,work1,north,south,east,west)

!---!-------------------------------------------------------------------
!---! Periodic/Neumann conditions for global domain boundaries
!---! Assumptions:  a *single* row of ghost cells (num_ghost_cells=1)
!---!
!---! work1 array has form (number_arrays,i_index,j_index)
!---!
!---! author Tony Craig, NCAR
!---!-------------------------------------------------------------------

      use ice_timers
      use ice_mpi_internal

      integer (kind=int_kind) :: i, j, nd, n
      real (kind=dbl_kind) :: work1(nd,imt_local,jmt_local)
      logical north,south,east,west
#ifdef _MPI
      integer (kind=int_kind) :: icnt,jcnt
     &,   status(MPI_STATUS_SIZE),request(4)
      real (kind=dbl_kind) :: workw(nd,jlo:jhi),worke(nd,jlo:jhi)
     &,  workn(nd,ilo-1:ihi+1),works(nd,ilo-1:ihi+1)
#endif

      call ice_timer_start(10) ! bound

#ifdef _MPI
      jcnt=(jhi-jlo+1)*nd
      icnt=(ihi-ilo+1+2*num_ghost_cells)*nd

      !-----------------------------------------------------------------
      ! west data to east data, west shift
      !-----------------------------------------------------------------
      if (east) then

      do j=jlo,jhi
      do n=1,nd
      workw(n,j)=work1(n,ilo,j)
      enddo
      enddo

      call MPI_SENDRECV(workw,jcnt,MPI_REAL8,nbr_west,my_task,
     &                  worke,jcnt,MPI_REAL8,nbr_east,nbr_east,
     &                  MPI_COMM_ICE,status,ierr)

      do j=jlo,jhi
      do n=1,nd
      work1(n,ihi+1,j)=worke(n,j)
      enddo
      enddo

      endif

      !-----------------------------------------------------------------
      ! east data to west data, east shift
      !-----------------------------------------------------------------
      if (west) then

      do j=jlo,jhi
      do n=1,nd
      worke(n,j)=work1(n,ihi,j)
      enddo
      enddo

      call MPI_SENDRECV(worke,jcnt,MPI_REAL8,nbr_east,my_task,
     &                  workw,jcnt,MPI_REAL8,nbr_west,nbr_west,
     &                  MPI_COMM_ICE,status,ierr)

      do j=jlo,jhi
      do n=1,nd
      work1(n,ilo-1,j)=workw(n,j)
      enddo
      enddo

      endif

      !-----------------------------------------------------------------
      ! north data to south data, north shift
      !-----------------------------------------------------------------
      if (south) then
      if (nbr_south.ne.-1) then
        call MPI_IRECV(works,
     &                 icnt,MPI_REAL8,nbr_south,nbr_south,
     &                 MPI_COMM_ICE,request(1),ierr)
      else

        do i=ilo-1,ihi+1
        do n=1,nd
        work1(n,i,jlo-1)=work1(n,i,jlo)
        enddo
        enddo

      endif
      if (nbr_north.ne.-1) then

        do i=ilo-1,ihi+1
        do n=1,nd
        workn(n,i)=work1(n,i,jhi)
        enddo
        enddo

        call MPI_ISEND (workn,
     &                 icnt,MPI_REAL8,nbr_north,my_task,
     &                 MPI_COMM_ICE,request(2),ierr)
      endif
      if (nbr_north.ne.-1) then
        call MPI_WAIT(request(2), status, ierr)
      endif
      if (nbr_south.ne.-1) then
        call MPI_WAIT(request(1), status, ierr)

        do i=ilo-1,ihi+1
        do n=1,nd
        work1(n,i,jlo-1)=works(n,i)
        enddo
        enddo

      endif
      endif

      !-----------------------------------------------------------------
      ! south data to north data, south shift
      !-----------------------------------------------------------------
      if (north) then
      if (nbr_north.ne.-1) then
        call MPI_IRECV(workn,
     &                 icnt,MPI_REAL8,nbr_north,nbr_north,
     &                 MPI_COMM_ICE,request(3),ierr)
      else

        do i=ilo-1,ihi+1
        do n=1,nd
        work1(n,i,jhi+1)=work1(n,i,jhi)
        enddo
        enddo

      endif
      if (nbr_south.ne.-1) then

        do i=ilo-1,ihi+1
        do n=1,nd
        works(n,i)=work1(n,i,jlo)
        enddo
        enddo

        call MPI_ISEND (works,
     &                 icnt,MPI_REAL8,nbr_south,my_task,
     &                 MPI_COMM_ICE,request(4),ierr)
      endif
      if (nbr_south.ne.-1) then
        call MPI_WAIT(request(4), status, ierr)
      endif
      if (nbr_north.ne.-1) then
        call MPI_WAIT(request(3), status, ierr)

        do i=ilo-1,ihi+1
        do n=1,nd
        work1(n,i,jhi+1)=workn(n,i)
        enddo
        enddo

      endif
      endif

#else
      !-----------------------------------------------------------------
      ! single domain
      !-----------------------------------------------------------------

      ! Periodic conditions
c$OMP PARALLEL DO PRIVATE(j)
      do j=jlo,jhi
      do n=1,nd
       work1(n,ilo-1,j) = work1(n,ihi,j)
       work1(n,ihi+1,j) = work1(n,ilo,j)
      enddo
      enddo

      ! Neumann conditions (POP grid land points)
c$OMP PARALLEL DO PRIVATE(i)
      do i=ilo-1,ihi+1
      do n=1,nd
        work1(n,i,jlo-1) = work1(n,i,jlo)
        work1(n,i,jhi+1) = work1(n,i,jhi)
      enddo
      enddo

#endif
      call ice_timer_stop(10)  ! bound

      end subroutine bound_nij

c=======================================================================

      end module ice_grid

c=======================================================================
