c $Id$
c=======================================================================
!---! Energy-conserving sea ice model
!---! Main routines for vertical thermodynamics
!---!     
!---! author C. M. Bitz
!---!     
!---! See Bitz, C.M., and W.H. Lipscomb, 1999: 
!---! An energy-conserving thermodynamic model of sea ice,
!---! J. Geophys. Res., 104, 15,669-15,677. 
!---!     
!---! See Bitz, C.M., M.M. Holland, A.J. Weaver, M. Eby:
!---! Simulating the ice-thickness distribution in a climate model
!---! accepted to J. Geophys. Res.
!---! (prepublications copy available upon request)
!---!     
!---! The author grants permission to the public to copy and use this
!---! software without charge, provided that this Notice and any statement
!---! of authorship are reproduced on all copies and any publications that
!---! result from the use of this software must (1) refer to the publications 
!---! listed above and (2) acknowledge the origin and author of the model.
!---! This software is without warranty, expressed or implied, and the
!---! author assumes no liability or responsibility for its use. 
c=======================================================================

      module ice_vthermo

      use ice_model_size
      use ice_domain 
      use ice_constants
      use ice_state

      implicit none

      real (kind=dbl_kind) ::
     &   saltz(nmax+1,ncat) ! salinity of each layer for each cat        (ppm)
     &,  tmelz(nmax  ,ncat) ! melting temper. of each layer for each cat   (C)

c=======================================================================

      contains

c=======================================================================

      subroutine init_thermo

!---!-------------------------------------------------------------------
!---! setup the salinity profile and the melting temperature
!---! for each layer
!---!-------------------------------------------------------------------

      use ice_itd

      integer (kind=int_kind) :: layer,nc,nsum
      real (kind=dbl_kind) ::  zn            ! normalized sea ice thickness

      ! number of ice layers in each thickness category
      ! it is essential that the number of layers in a given 
      ! category be an integer multiple of the number 
      ! of layers in a lower category
      ! factors of 2 work well
      if (ncat.eq.1) then
      nilay(1) = 4
      elseif (ncat.le.3) then
      nilay(1) = 2
      do nc=2,ncat
         nilay(nc) = 4
      enddo
      elseif (ncat.le.5) then
      nilay(1) = 2
      nilay(2) = 2
      do nc=3,ncat
         nilay(nc) = 4
      enddo
      else
      nilay(1) = 2
      nilay(2) = 2
      nilay(3) = 2
      do nc=4,ncat
         nilay(nc) = 4
      enddo
      endif

      nsum=0
      do nc=1,ncat
         nsum=nsum+nilay(nc)
      enddo
      if (nsum.ne.ntilay) stop 'the sum of nilay must be ntilay'

      ! salinity and melting temperature profile
      do nc=1,ncat
         do layer=1,nilay(nc)
            zn=(layer-p5)/nilay(nc)
            saltz(layer,nc)=(saltmax/c2)*
     $          (c1-cos(pi*zn**(0.407_dbl_kind/(0.573_dbl_kind+zn))))
c            saltz(layer,nc)=saltmax ! for isosaline ice
         enddo
         saltz(nilay(nc)+1,nc)=saltmax
         do layer=1,nilay(nc)
            tmelz(layer,nc)=-saltz(layer,nc)*depressT
         enddo
c         if (my_task.eq.master_task) then
c           write (6,*) '   salt profile, cat ',nc
c           write (6,*) (saltz(layer,nc),layer=1,nilay(nc)+1)
c           write (6,*) '   melt temp, cat ',nc
c           write (6,*) (tmelz(layer,nc),layer=1,nilay(nc))
c         endif
      enddo

      end subroutine init_thermo

c=======================================================================

      subroutine thermo_vertical(i,j,hin,hsn,tiz,Fbot,
     &         dhi1,dhin,dhsn,hfatm,hfocn,qin)

!---!-------------------------------------------------------------------
!---! heat budget over open water and ice 
!---! NOTE the wind stress is also computed here for later use
!---!-------------------------------------------------------------------

      use ice_flux
      use ice_albedo
      use ice_atmo
      use ice_history
      use ice_calendar
      use ice_itd
      use ice_dh
      use ice_tstm

      integer (kind=int_kind), intent(in) :: i, j
      real (kind=dbl_kind), intent(in) ::
     &   Fbot

      real (kind=dbl_kind), intent(inout) ::
     &   hin(ncat)          ! ice thickness for each cat        (m)
     &,  hsn(ncat)          ! snow thickness for each cat       (m)
     &,  tiz(0:nmax,ncat)  ! temp of each layer for each cat   (C)

      real (kind=dbl_kind), intent(out) ::
     &   dhi1        ! melt at bottom and top plus sublimation  (m)
     &,  dhin(ncat)   ! ice thickness change (m)
     &,  dhsn(ncat)   ! snow thickness change (m)
     &,  hfatm       ! heat flx  to ice-snow from atm  (J/m**2)
     &,  hfocn       ! heat flx  to ice-snow from ocn  (J/m**2)
     &,  qin(nmax,ncat)  ! enthalpy per unit volume     (J/m**3)

      integer (kind=int_kind) :: nc, layer, k

      real (kind=dbl_kind) ::
     &   Trefn       ! air tmp rfrnc level                  (K)
     &,  dhib        ! melt at bottom                       (m)
     &,  dhit        ! ice melt at top                      (m)
     &,  subi        ! ice sublimation                      (m)
     &,  flwupn       ! upwd lw emitted heat flx        (W/m**2)
     &,  flwdabs      ! down lw absorbed heat flx       (W/m**2)
     &,  flhn        ! latent   heat flx               (W/m**2)
     &,  fshn        ! sensible heat flx               (W/m**2)
     &,  strxn       ! air/ice zonal  strss,           (N/m**2)
     &,  stryn       ! air/ice merdnl strss,           (N/m**2)
     &,  frshn       ! fresh water flux to ocean (kg/m2/s)
     &,  evapn      ! evaporation (kg/m2/s)
     &,  dflwn      ! deriv. upwd lw  heat flx        (W/m**2)
     &,  dflhn       ! deriv. latent   heat flx        (W/m**2)
     &,  dfshn       ! deriv. sensible heat flx        (W/m**2)
     &,  fswtop       ! net sw into ice/snow            (W/m**2)
     &,  fswtopv      ! fswtop in vis (wvlngth < 700nm)  (W/m**2)
     &,  fswtopn      ! fswtop in nir (wvlngth > 700nm)  (W/m**2)
     &,  swbot       ! sw radiation through ice bot    (W/m**2)
     &,  asnow      ! 1 - patchy snow frac
     &,  fnet        ! net flx at ice/snow top srf     (N/m**2)
     &,  condb       ! conductive flx at ice bot       (N/m**2)
     &,  tbot(ncat)        ! temp of ice bottom for each cat   (C)
     &,  dhs         ! snow melt                            (m)
     &,  subs        ! snow sublimation                     (m)
     &,  dhif        ! ice thickness change from flooding   (m)
     &,  dhsf        ! snow thickness change from flooding   (m)
     &,  Focn        ! actual ocn/ice heat flx         (W/m**2)
     &,     dssqdt ! derivative of ssq wrt Ti (kg/kg/K)
     &,     delt   ! potential T difference   (K)
     &,     delq   ! humidity difference      (kg/kg)
     &,  rdn_in  ! initial value for rdn in stability routine

         Tref(i,j) = c0 
         dhi1 = c0

         do nc = 1,ncat

           dhi1 = c0
           dhin(nc) = c0
           dhsn(nc) = c0
           lhcoef(i,j,nc) = c0
           shcoef(i,j,nc) = c0

           if (ain(nc).gt.puny) then

      !-----------------------------------------------------------------
      ! compute air to ice heat, momentum, radiative and water fluxes 
      !-----------------------------------------------------------------

           rdn_in = vonkar/log(zref/iceruf)
           call stability( i,j,nc, Tsfn(nc), rdn_in,
     $         strxn   ,stryn   ,Trefn, dssqdt, delt, delq)

           call ice_sfc_flux(i,j,nc, dssqdt, delt, delq,
     &      flwdabs,flwupn, fswtop,fswtopv,fswtopn,
     &      fshn,flhn,dflhn,dfshn,dflwn)

      !-----------------------------------------------------------------
      ! add snowfall, which is in kg/m2/s 
      !-----------------------------------------------------------------
            ! snow lands on ice no matter what its temperature
            hsn(nc) =  hsn(nc)+(Fsnow(i,j)/rhos)*dt

            ! 1 - snow covered area fraction
            asnow = c1-hsn(nc)/(hsn(nc) + snowpatch)

      !-----------------------------------------------------------------
      ! solve heat equation, updating Tiz
      !-----------------------------------------------------------------
            call tstm( dt, Tmelz(:,nc), saltz(:,nc), Tf(i,j)
     $                , ain(nc), hin(nc), hsn(nc), fswtop
     $                , fswtopv, fswtopn
     $                , flwdabs, dflwn, dflhn, dfshn  
     $                , asnow,  nilay(nc), tbot(nc)
     $                , swbot, Fnet, condb, Tsfn(nc), Tiz(0:,nc)
     $                , flwupn, flhn, fshn)

      !-----------------------------------------------------------------
      ! compute ice/snow thickness changes
      !-----------------------------------------------------------------
            call dh  (  dt, saltz(:,nc), Tiz(0:,nc)
     $                , tbot(nc), hin(nc), hsn(nc), Fbot
     $                , fnet, condb, flhn, nilay(nc)
     $                , dhib, dhit, dhs, subi
     $                , subs, dhif, dhsf, qin(:,nc), Focn, i,j)

            dhi1 = dhit + dhib + subi  ! used only for ncat=1 case
            dhin(nc) = dhit + dhib + subi + dhif
            dhsn(nc) = dhs  + subs + dhsf

      !-----------------------------------------------------------------
      ! aggregate fluxes for coupler
      !-----------------------------------------------------------------
            frshn =   - ((dhs+dhsf)*rhos + (dhit+dhib+dhif)*rhoi)/dt
            evapn = (rhos*subs+rhoi*subi)/dt 
            call merge_fluxes(i,j,nc,frshn,evapn,
     &        fshn,flhn,flwupn,swbot,Focn,trefn,strxn,stryn)

      !-----------------------------------------------------------------
      ! net heat fluxes
      ! to check conservation later
      !-----------------------------------------------------------------
            hfatm = hfatm + (fswtop + flwdabs + fshn + flwupn 
     $           +(rLvs*subs+rLvi*subi)/dt-Fsnow(i,j)*Lfus)
     $           *ain(nc)
            hfocn = hfocn + Focn*ain(nc)

      !-----------------------------------------------------------------
      ! history and diagnostics
      !-----------------------------------------------------------------
            ! history
            growb (i,j) = growb (i,j) + max(c0,dhib)*ain(nc) ! ice growth
            snoice(i,j) = snoice(i,j) + dhif*ain(nc)         ! snow-ice conv
            meltb(i,j) = meltb(i,j) - min(c0,dhib)*ain(nc)   ! basal melt >0
            meltt(i,j) = meltt(i,j) - dhit*ain(nc)           ! top ice melt >0
            sabs(i,j) = sabs(i,j) + fswtop*ain(nc)           ! absorbed solar

           else  ! ain(nc) = 0
            Tsfn(nc) = Tf(i,j)  ! open water
           endif
          enddo                 ! loop over bins

          ! diagnostic only
          Fhdiag(i,j) = sabs(i,j) + Flw(i,j)*aice(i,j)

      end subroutine thermo_vertical

c=======================================================================

      end module ice_vthermo

c=======================================================================

