c $Id$
c=======================================================================
!---! Energy-conserving sea ice model
!---! Driver for thermodynamics and associated changes to the itd
!---!
!---! author C. M. Bitz
!---!
!---! See Bitz, C.M., and W.H. Lipscomb, 1999: 
!---! An energy-conserving thermodynamic model of sea ice,
!---! J. Geophys. Res., 104, 15,669-15,677. 
!---!     
!---! See Bitz, C.M., M.M. Holland, A.J. Weaver, M. Eby:
!---! Simulating the ice-thickness distribution in a climate model
!---! accepted to J. Geophys. Res.
!---! (prepublications copy available upon request)
!---!
!---! The author grants permission to the public to copy and use this
!---! software without charge, provided that this Notice and any statement
!---! of authorship are reproduced on all copies and any publications that
!---! result from the use of this software must (1) refer to the publications 
!---! listed above and (2) acknowledge the origin and author of the model.
!---! This software is without warranty, expressed or implied, and the
!---! author assumes no liability or responsibility for its use. 
c=======================================================================
!---! code heavily modified by Elizabeth C. Hunke, LANL
c=======================================================================

      module ice_therm_driver

      use ice_kinds_mod
      use ice_model_size
      use ice_constants
      use ice_state
      use ice_flux
      use ice_diagnostics
      use ice_calendar

      implicit none

      real (kind=dbl_kind), private, save ::
     &   Fnewice(ilo:ihi,jlo:jhi) ! heat flx to open water          (W/m**2)
     &,  Rsidmlt(ilo:ihi,jlo:jhi) ! fraction of ice that melts from side
     &,  Fatm(ilo:ihi,jlo:jhi)    ! heat flx  to ice-snow from atm  (J/m**2)
     &,  einit(ilo:ihi,jlo:jhi)   ! initial energy in ice/snow      (J/m**2)
     &,  dhice1(ilo:ihi,jlo:jhi)   ! ice thickness change, 1-category case (m)
     &,  hicen(ncat,ilo:ihi,jlo:jhi)   ! ice thickness (m)
     &,  hsnon(ncat,ilo:ihi,jlo:jhi)   ! snow thickness (m)
     &,  dhicen(ncat,ilo:ihi,jlo:jhi)   ! ice thickness change (m)
     &,  dhsnon(ncat,ilo:ihi,jlo:jhi)   ! snow thickness change (m)
     &,  qicen(nmax,ncat,ilo:ihi,jlo:jhi)  ! enthalpy per unit volume (J/m**3)

c=======================================================================

      contains

c=======================================================================

      subroutine thermo_rates

!---!-------------------------------------------------------------------
!---! compute thermodynamic growth rates and atmospheric fluxes
!---!-------------------------------------------------------------------

      use ice_domain
      use ice_timers
      use ice_vthermo
      use ice_grid
      use ice_itd
      use ice_init
      use ice_ocean

      integer (kind=int_kind) :: i, j

      real (kind=dbl_kind) ::
     &   Fbot        ! heat flx to ice bottom          (W/m**2)
     &,  tiz(0:nmax,ncat)  ! temp of each layer for each cat   (C)

      real (kind=dbl_kind)
     &  Frsh_init       ! initial Fresh, use to get correct budgets
     &, Fh_init         ! initial Fhnet, use to get correct budgets
     &, Fswthru_init    ! initial Fswthru

      call ice_timer_start(11)  ! pre-cpl
      call ice_timer_start(4)  ! 1  model
      do j=jlo,jhi
       do i=ilo,ihi

      !-----------------------------------------------------------------
      ! initialize 
      !-----------------------------------------------------------------
        Frsh_init = Fresh(i,j)	
        Fh_init   = Fhnet(i,j)
        Fswthru_init = Fswthru(i,j)
        call init_flux_atm(i,j)               ! atm fluxes
        call init_column_diags(i,j,Fatm(i,j)) ! init diagnostic var

        call ice_timer_start(5)  ! thermodynamics
        if (tmask(i,j)) then    ! ocean T cells


          call mixed_layer(i,j)  ! get ocean forcing if desired


          call to_column(i,j)      ! define 1  state variables

          ! initialize lateral growth/melt (thermo_vertical needs Fbot)
          call init_frzmlt(i,j,Fbot,Fnewice(i,j),Rsidmlt(i,j))

      !-----------------------------------------------------------------
      ! vertical thermodynamics: growth rates and fluxes for coupling
      !-----------------------------------------------------------------
          call init_vertical_profile(i,j,einit(i,j),hicen(:,i,j)
     &            ,hsnon(:,i,j),tiz)

          if (aice(i,j).gt.puny) then
            call thermo_vertical(i,j,hicen(:,i,j),hsnon(:,i,j),tiz,Fbot,
     &      dhice1(i,j),dhicen(:,i,j),dhsnon(:,i,j),
     &      Fatm(i,j),Fhocn(i,j),qicen(:,:,i,j))

            ! divide fields by aice
            strairxT(i,j)   = strairxT(i,j)   /aice(i,j)
            strairyT(i,j)   = strairyT(i,j)   /aice(i,j)
            Tref(i,j)      = Tref(i,j)      /aice(i,j)  

            ! history and diagnostics
            Frsh_hist(i,j) = Fresh(i,j) - Frsh_init  ! use for diagnostics
            Foht_hist(i,j) = Fhnet(i,j) - Fh_init    ! use for diagnostics
            Fswthru_hist(i,j) = Fswthru(i,j) - Fswthru_init ! use for diagnostics
            Fhocn(i,j) = Fhocn(i,j) + Fswthru_hist(i,j) 


            ! update mixed layer with heat from ice
            sst(i,j) = sst(i,j)+(Fhnet(i,j)+Fswthru(i,j))*dt/cphm 
            sst_mixedlayer(i,j) = sst(i,j)  ! save value

          endif

          call from_column(i,j) ! update Tsfc
        endif ! tmask
        call ice_timer_stop(5)   ! thermodynamics
       enddo
      enddo
      call ice_timer_stop(4)  ! 1  model
      call ice_timer_stop(11)  ! pre-cpl

      end subroutine thermo_rates

c=======================================================================

      subroutine thermo_itd

!---!-------------------------------------------------------------------
!---! changes to ice thickness distribution associated with 
!---! thermodynamic growth rates, including lateral growth/melt
!---! NOTE:  ocean fluxes are initialized here
!---!-------------------------------------------------------------------

      use ice_domain
      use ice_timers
      use ice_history
      use ice_grid
      use ice_itd
      use ice_itd_linear
      use ice_init

      integer (kind=int_kind) :: i, j, nc, layer

      real (kind=dbl_kind) ::
     &   hin(ncat)          ! ice thickness for each cat        (m)
     &,  hsn(ncat)          ! snow thickness for each cat       (m)

      call ice_timer_start(4)  ! 1  model
      do j=jlo,jhi
       do i=ilo,ihi

      !-----------------------------------------------------------------
      ! initialize 
      !-----------------------------------------------------------------
        aice_init(i,j) = aice(i,j)              ! use for correct diagnostics
        call init_flux_ocn(i,j)                 ! ocean fluxes for coupler
        Fresh(i,j) = Frain(i,j)*aice(i,j)       ! rain drains thru to ocn

        if (tmask(i,j)) then
         call to_column(i,j)
         call ice_timer_start(5)   ! thermodynamics

      !-----------------------------------------------------------------
      ! update ice state
      !-----------------------------------------------------------------
         do nc = 1,ncat
           if (ain(nc).gt.puny) then
             hin(nc) = hicen(nc,i,j) + dhicen(nc,i,j)
             hsn(nc) = hsnon(nc,i,j) + dhsnon(nc,i,j)
             if (hin(nc).lt.puny) ain(nc) = 0.    !!!!!!!!!!! check me
             vin(nc) = hin(nc)*ain(nc)
             vsn(nc) = hsn(nc)*ain(nc)
             do layer = 1,nilay(nc)
               ein(layer,nc) = qicen(layer,nc,i,j)*vin(nc)/nilay(nc)
             enddo
           else
             hin(nc) = c0
             hsn(nc) = c0

             vin(nc) = c0
             vsn(nc) = c0
             do layer = 1,nilay(nc)
               ein(layer,nc) = c0
             enddo
           endif
         enddo                 ! loop over bins

         call ice_timer_stop(5)   ! thermodynamics

      !-----------------------------------------------------------------
      ! ice thickness redistribution
      !-----------------------------------------------------------------
         call ice_timer_start(7)  ! category conversions (advection in h)
         if (kitd.eq.1) then
           call linear_itd(i,j,dhicen(:,i,j),hicen(:,i,j),hin,hsn)
         endif
         call ice_timer_stop(7)   ! category conversions 

      !-----------------------------------------------------------------
      ! frazil ice growth / lateral melt
      !-----------------------------------------------------------------
         call ice_timer_start(5)  ! thermodynamics
         call lateral_growth_melt(i,j,Fnewice(i,j),Rsidmlt(i,j),hin)
         call ice_timer_stop(5)   ! thermodynamics

      !-----------------------------------------------------------------
      ! ice thickness redistribution
      !-----------------------------------------------------------------
         call ice_timer_start(7)  ! category conversions (advection in h)
         if (ncat.eq.1) then  
           call reduce_area(i,j,dhice1(i,j),hin,hsn)
         else
           call rebin_ice(i,j)
         endif
         call ice_timer_stop(7)   ! category conversions 

      !-----------------------------------------------------------------
      ! update cell values 
      !-----------------------------------------------------------------
         call aggregate_pt(i,j)
         ai0 = c1-aice(i,j)

      !-----------------------------------------------------------------
      ! check conservation
      !-----------------------------------------------------------------
         ! net ocean-ice heat flux 
         Fhocn(i,j) = Fhocn(i,j) + Fhnet(i,j) + Fnewice(i,j) 
         call conservation_check(i,j,einit(i,j),Fatm(i,j),Fhocn(i,j))

      !-----------------------------------------------------------------
      ! reload variables
      !-----------------------------------------------------------------
         call from_column(i,j)
         
      !-----------------------------------------------------------------
      ! update history and diagnostic fields
      !-----------------------------------------------------------------
         ! ice concentration & volume tendency
         daidtt(i,j) = (aice(i,j)-daidtt(i,j))/dt
         daidtd(i,j) = aice(i,j) ! temporarily used for initial conc.
         dvidtt(i,j) = (vice(i,j)-dvidtt(i,j))/dt
         dvidtd(i,j) = vice(i,j) ! temporarily used for initial volume
         ! fresh water and heat fluxes over current timestep
         Frsh_hist(i,j) = Frsh_hist(i,j) + Fresh(i,j)
         Foht_hist(i,j) = Foht_hist(i,j) + Fhnet(i,j)

         else
           Fhocn(i,j) = c0  ! for ocean mixed layer model
         endif                   ! tmask

      call check_state(i,j) 

       enddo
      enddo
      call ice_timer_stop(4)   ! 1  model

      end subroutine thermo_itd

c=======================================================================

      subroutine init_column_diags(i,j,hfatm)

!---!-------------------------------------------------------------------
!---! initialize diagnostic and history variables
!---!-------------------------------------------------------------------

      use ice_history

      integer (kind=int_kind) :: i, j
      real (kind=dbl_kind), intent(out) ::
     &   hfatm       ! heat flx  to ice-snow from atm  (J/m**2)

        ! for internal calculations and/or coupling
        hfatm = c0
        Fhocn  (i,j) = c0
        ! internal diagnostics
        Fhdiag (i,j) = c0
        ! diagnostics for history
        growb  (i,j) = c0
        frazil (i,j) = c0
        snoice (i,j) = c0
        meltl  (i,j) = c0
        meltb  (i,j) = c0
        meltt  (i,j) = c0
        sabs   (i,j) = c0
        dvidtt (i,j) = vice(i,j)
        dvidtd (i,j) = c0
        daidtt (i,j) = aice(i,j) ! temporarily used for initial conc.
        daidtd (i,j) = c0
        Frsh_hist(i,j) = c0
        Foht_hist(i,j) = c0
        Fswthru_hist(i,j) = c0

      end subroutine init_column_diags

c=======================================================================

      subroutine init_vertical_profile(i,j,ei0,hin,hsn,tiz)

!---!-------------------------------------------------------------------
!---! initialize the vertical profile of ice properties
!---!-------------------------------------------------------------------

      use ice_vthermo
      use ice_itd

      integer (kind=int_kind), intent(in) :: i,j

      real (kind=dbl_kind), intent(out) :: 
     &   ei0       ! initial energy in ice/snow      (J/m**2)
     &,  hin(ncat)          ! ice thickness for each cat        (m)
     &,  hsn(ncat)          ! snow thickness for each cat       (m)
     &,  tiz(0:nmax,ncat)  ! temp of each layer for each cat   (C)

      integer (kind=int_kind):: nc, layer
      real (kind=dbl_kind) ::
     &      T1, T2     ! two solutions of the quadratic eq    (C)
     &,     q, B, C   ! variables to help solve quad. eq.
     &,     B_2, root
     &,  qin(nmax,ncat)  ! enthalpy per unit volume     (J/m**3)

          ei0 = c0
          do nc = 1,ncat

          hin(nc) = c0
          hsn(nc) = c0
          do layer = 1,nilay(nc)
            tiz(layer,nc) = c0
          enddo

           if (ain(nc).gt.puny) then

          do layer = 1,nilay(nc)
            qin(layer,nc) = ein(layer,nc)*nilay(nc)/vin(nc)
            ei0 = ei0+ein(layer,nc)
          enddo

      !-----------------------------------------------------------------
      !     ice and snow thickness
      !-----------------------------------------------------------------
            hin(nc) = vin(nc)/ain(nc)
            hsn(nc) = vsn(nc)/ain(nc)
      !-----------------------------------------------------------------
      !     vertical temperature profile
      !-----------------------------------------------------------------
      ! compute the midpoint temperature from the 
      ! energy of melting for each layer
      ! by solving the quadratic equation
      !-----------------------------------------------------------------
      do layer = 1,nilay(nc)
        q = qin(layer,nc)+rLfi-rcpidepressT*saltz(layer,nc)
        B = -q/rcpi
        C = -rLfidepressT*saltz(layer,nc)/rcpi
        B_2  = B/c2
        root = sqrt(B_2*B_2-C)
c        T1   = -B_2+root
        T2   = -B_2-root
        tiz(layer,nc) = T2
      enddo
            !snow temperature needs a reasonable starting value
            tiz(0,nc) = tiz(1,nc) ! snow temp is diagnostic
           endif

          enddo ! ncat

         ei0 = ei0-vsno(i,j)*rLfs

      end subroutine init_vertical_profile

c=======================================================================

      subroutine init_frzmlt(i,j,Fbot,Fnew,Rside)

!---!-------------------------------------------------------------------
!---! initialize ocean-ice heat fluxes  bottom and lateral
!---! Assuming frzmlt is per grid box area
!---!-------------------------------------------------------------------

      use ice_itd

      integer (kind=int_kind), intent(in) :: i, j
      real (kind=dbl_kind), intent(out) ::
     &   Fbot        ! heat flx to ice bottom,         (W/m**2)
     &,  Fnew        ! heat flx to open water,         (W/m**2)
     &,  Rside       ! fraction of ice that melts from side

      real (kind=dbl_kind) ::
     &   Fsid        ! heat flx to side of ice (neg)    (W/m**2)
     &,  ustar       ! skin friction velocity for Fbot (m/s)
     &,  htot        ! mean ice thickness
     &,  frcbot      ! fraction of available melt heat for bottom
     &,  frcsid      ! fraction of available melt heat for side
     &,  Fbotmx      ! maximum heat available for bottom melt
     &,  Fsidmx      ! maximum heat available for side melt
     &,  etot
     &,  deltaT

      real (kind=dbl_kind), parameter ::
     &  Rfactor =  0.68_dbl_kind     ! shortwave penetration factor
     &,  zeta1  =  1.20_dbl_kind       ! shortwave e-folding depth
     &,  zeta2  = 28.00_dbl_kind     ! shortwave e-folding depth

      integer (kind=int_kind) :: nc, layer

      ! Parameters for basal and lateral heat flx 
      ! 0.006 = unitless param for basal heat flx ala McPhee and Maykut
      real (kind=dbl_kind), parameter :: 
     &   cpchr =-cp_ocn*rhow*0.006_dbl_kind
      ! params for lateral heat flx ala Maykut and Steel
      real (kind=dbl_kind), parameter :: 
     &   m1 = 1.6e-6_dbl_kind       ! (m/s/deg**m2)
     &,  m2 = 1.36_dbl_kind         ! unitless
     &,  floe_size = 300.0_dbl_kind       ! floe diameter (m)
      real (kind=dbl_kind) :: pi_eta
      pi_eta = pi/0.66_dbl_kind ! unitless 

          if (frzmlt(i,j) .ge. 0.) then     
      !-----------------------------------------------------------------
      ! freezing conditions 
      !-----------------------------------------------------------------
            Fbot = c0
            Fnew = frzmlt(i,j)
            Rside = c0

          else

      !-----------------------------------------------------------------
      ! melting conditions
      !-----------------------------------------------------------------
      ! Compute fraction of heat available to melt side (frcsid)
      ! and that available to melt bottom (frcbot), and limit the
      ! amounts available according to frcsid*frzmlt and frcbot*frzmlt
      ! respectively
      !-----------------------------------------------------------------

           if (aice(i,j).gt.c0) then

            htot   = vice(i,j)/aice(i,j)
            frcbot = Rfactor*exp(-htot/zeta1) + 
     $               (c1-Rfactor)*exp(-htot/zeta2)
            frcsid = c1 - frcbot
            Fbotmx = frcbot*frzmlt(i,j)
            Fsidmx = frcsid*frzmlt(i,j)

            ! use boundary layer theory for Fbot

            deltaT = max((sst(i,j)-Tf(i,j)),c0) 

            ! strocnx has units N/m**2 so strocnx/rho has units m**2/s**2 
            ustar = sqrt(   sqrt(strocnxT(i,j)**2
     $           +strocnyT(i,j)**2)  /rhow    )
            ustar = max(ustar,.001_dbl_kind)

            Fbot = cpchr*deltaT*ustar  ! < 0
            Fbot = max(Fbot,Fbotmx)    ! Fbotmx < Fbot < 0

            Fnew = c0

            ! allow for side melt

            etot = c0
            do nc = 1,ncat
             if (ein(1,nc).lt.-puny) then
              etot = etot + vsn(nc)*rLfs
              do layer=1,nilay(nc)
               etot = etot - ein(layer,nc) ! positive
              enddo
             endif
            enddo
            etot   = etot
            Fsid   = etot*pi_eta/floe_size*m1*deltaT**m2
            Fsid   = max(Fsid,Fsidmx)
            Rside  = Fsid*dt/etot
            Rside  = min(Rside,c1) ! positive

           else  ! aice = 0
            Fbot = c0
            Fnew = c0
            Rside = c0
           endif ! aice
          endif ! sgn(frzmlt)

      end subroutine init_frzmlt

c=======================================================================

      subroutine lateral_growth_melt(i,j,Fnew,Rside,hin)

!---!-------------------------------------------------------------------
!---! frazil ice growth / lateral melt
!---!-------------------------------------------------------------------

      use ice_vthermo
      use ice_history
      use ice_itd

      integer (kind=int_kind), intent(in) :: i,j

      real (kind=dbl_kind), intent(in) ::
     &   Fnew        ! heat flx to open water          (W/m**2)
     &,  Rside       ! fraction of ice that melts from side

      real (kind=dbl_kind), intent(inout) ::
     &   hin(ncat)          ! ice thickness for each cat        (m)

      integer (kind=int_kind) :: layer, nc, nadd
      real (kind=dbl_kind) ::
     &    vio         ! volume of new ice                    (m)
     &,   Tnew        ! temp of top srf of new ice           (C)
     &,   h0prime     ! new ice thickness                            (m)
     &,   aioprime    ! new ice fraction 
     &,   ai1prime    ! sum of cat 1 and new ice fraction 
     &,   Tm          ! ice melting temperature at salinity Si (C)
     &,   Ti          ! frazil ice formation temperature (sst, C)
     &,   qf          ! new ice energy of melting per unit mass (J/kg)
     &,   qio         ! new ice energy of melting per unit volume (J/m**3)
     &,   eio         ! new ice energy of melting                 (J/m**2)
     &,   tmp         ! dummy temp                                   (C)
     &,   hnew        ! thickness of new ice                 (m)
     &,   frc
     &,   etot
      logical (kind=log_kind) :: addice

          vio = c0
          if (Fnew.gt.0.) then 

      !-----------------------------------------------------------------
      ! grow frazil ice 
      !-----------------------------------------------------------------

cxc the following is how I (cc) would prefer to grow ice in the lead
cxc but csm ocean computes frazil ice heat and salt fluxes, so 
cxc we must assume a fixed enthalpy that is equivalent
cxc in both models
cxc           Tnew = min(sst(i,j),p5*(tair(i,j)-Tffresh)) ! in C
cxc           tmp = p5*(sst(i,j)+Tnew) ! in C
cxc           qio = energ(tmp,salnew)
cxc           Fhnet(i,j) = Fhnet(i,j) + Fnew

           Tnew = min(sst(i,j),p5*(Tair(i,j)-Tffresh)) ! in C
           qio = -rhoi*Lfus

           vio = -Fnew*dt/qio
           frazil(i,j) = frazil(i,j) + vio*rhoi/dt     ! for history

           if ( ai0 .gt. puny ) then
           !  add lead ice to cat 1 by first making the lead ice
           !  at least as thick as hin_max(0) and then finding the frac of 
           !  the lead that would be covered by hin_max(0) thick ice
            hnew = vio/ai0
            H0prime  = max(hi_min,hnew)
            aioprime = vio/H0prime 
            nadd = 1
           else  ! ai0 < puny
           ! if ocean wants to grow frazil ice, but there is no open 
           ! water, add the volume to the first category with 
           ! available ice but do not increase the area
            aioprime = c0
            addice = .false.
            do nc=1,ncat
              if ((.not.addice).and.(ain(nc).gt.0.)) then
                nadd = nc
                addice = .true.
              endif
            enddo
           endif  ! ai0

           ai1prime = aioprime + ain(nadd)
           vin(nadd) = vio + vin(nadd)
           Tsfn(nadd) = (Tsfn(nadd)*ain(nadd)
     $          +Tnew*aioprime)/ai1prime
           ain(nadd) = ai1prime
           hin(nadd) = vin(nadd)/ai1prime
           eio = vio*qio/nilay(nadd) ! divide nrg of melt into nilay(1)
           do layer = 1,nilay(nadd)
             ein(layer,nadd) = eio+ein(layer,nadd)
           enddo

          elseif (aice(i,j).gt.puny) then 

      !-----------------------------------------------------------------
      ! melt laterally (Fnew >= 0)
      !-----------------------------------------------------------------

           do nc = 1,ncat
            if (ein(1,nc).lt.-puny) then
             etot = vsn(nc)*rLfs
             do layer=1,nilay(nc)
              etot = etot - ein(layer,nc) ! positive
             enddo

             frc = c1- Rside     ! fraction that is unmelted

             Fhnet(i,j) = Fhnet(i,j) - Rside*etot/dt
             Fresh(i,j) = Fresh(i,j) + (vsn(nc)*Rside*rhos
     $                               +  vin(nc)*Rside*rhoi)/dt

             ! lateral ice melt for history (positive)
             meltl(i,j) = meltl(i,j) + vin(nc)*Rside

             vin(nc) = vin(nc)*frc
             vsn(nc) = vsn(nc)*frc
             ain(nc) = ain(nc)*frc
             do layer = 1,nilay(nc)
              ein(layer,nc) = ein(layer,nc)*frc
             enddo

            endif ! ein
           enddo  ! nc

          endif   ! Fnew

      end subroutine lateral_growth_melt

c=======================================================================

      subroutine reduce_area(i,j,dhi1,hin,hsn)

!---!-------------------------------------------------------------------
!---! Reduce area when ice melts for special case of ncat=1
!---!
!---! Use CSM 1.0-like method of reducing ice area
!---! when melting occurs: assume only half the ice volume
!---! change goes to thickness decrease, the other half
!---! to reduction in ice fraction
!---!-------------------------------------------------------------------

      use ice_kinds_mod
      use ice_constants
      use ice_flux  !! not necessary if rebin is called instead
      use ice_state
      use ice_itd

      integer (kind=int_kind), intent(in) :: i,j

      real (kind=dbl_kind), intent(in) ::
     &  dhi1        ! melt at bottom and top plus sublimation  (m)

      real (kind=dbl_kind), intent(inout) ::
     &   hin(ncat)          ! ice thickness for each cat        (m)
     &,  hsn(ncat)          ! ice thickness for each cat        (m)

      integer (kind=int_kind) :: nc

      real (kind=dbl_kind) ::
     &   hi0         ! current hi for ice fraction adjustment
     &,  dai0        ! change in aice for ice fraction adjustment

      !-----------------------------------------------------------------
      ! make sure thickness of cat 1 is at least hin_max(0)
      !-----------------------------------------------------------------
          if ((hin(1).lt.hin_max(0)).and.(hin_max(0).gt.0.)) then
           ain(1) = vin(1)/hin_max(0)
           hin(1) = hin_max(0)
          endif

            if( ain(1) .gt. 0. ) then 
              if( dhi1 .lt. 0.0 ) then  
                hi0        = vin(1)/ain(1)
                dai0       = (vin(1)/(hi0-(dhi1/c2))) - ain(1)
                ain(1) = ain(1) + dai0 
              endif
            endif

      end subroutine reduce_area

c=======================================================================

      subroutine conservation_check(i,j,ei0,hfatm,hfocn)

!---!-------------------------------------------------------------------
!---! make sure ice properties are properly conserved
!---!-------------------------------------------------------------------

      use ice_itd

      integer (kind=int_kind), intent(in) :: i, j

      real (kind=dbl_kind), intent(in) ::
     &   hfatm       ! heat flx  to ice-snow from atm  (J/m**2)
     &,  hfocn       ! heat flx  to ice-snow from ocn  (J/m**2)
     &,  ei0       ! initial energy in ice/snow      (J/m**2)

      real (kind=dbl_kind) ::
     &   efinl       ! final energy in ice/snow        (J/m**2)
     &,  ediff       ! energy difference in ice/snow   (J/m**2)
     &,  erat        ! mass ratio to ensure energy conservation

      real (kind=dbl_kind), parameter ::
     &   maxrat = 1.50_dbl_kind    ! maximum ratio allowed for correction
c     &,  e_errmax = 5.0e2_dbl_kind ! error tolerance for energy diff (J)
     &,  e_errmax = 3.0e3_dbl_kind ! error tolerance for energy diff (J)

      integer (kind=int_kind) :: layer, nc
     &,  nunit    ! for diagnostic prints  

      logical (kind=log_kind) ::
     &   ecfrc    ! if true, force energy conservation

      data ecfrc / .false. /

         efinl = c0
         do nc = 1,ncat
          do layer = 1,nilay(nc)
           efinl = efinl+ein(layer,nc)
          enddo
         enddo
         efinl = efinl-vsno(i,j)*rLfs
         ediff = (efinl-ei0)-(hfatm-hfocn)*dt 

         if (abs(ediff).gt.e_errmax) then
           nunit = 30 + my_task
           write(nunit,998) i,j,ediff,(hfatm-hfocn)*dt*1.e-9,
     &                       (efinl-ei0)*1.e-9
 998       format(' big enrgy diff ice model i j= ',2(i3,1x),
     $            3(1pe11.5,1x))
            write(6,*) 'dt,ediff,e_errmax',dt,ediff,e_errmax
            write(6,*) 'efinl,ei0',efinl,ei0
            write(6,*) 'efinl should be',ei0+(hfatm-hfocn)*dt 
            write(6,*) 'equiv flux diff',ediff/dt
            write(6,*) 'hfatm,hfocn',hfatm,hfocn
            write(6,*) 'efinl+vsno*rLfs',efinl+vsno(i,j)*rLfs
            write(6,*) 'should be',ei0+(hfatm-hfocn)*dt+vsno(i,j)*rLfs
            call print_state(' -----ice state-----',i,j)
c            stop
         endif

      !-----------------------------------------------------------------
      ! force energy conservation point by point if desired; 
      ! i.e. make internal energy changes consistent with net energy fluxes
      ! on the ice/snow system; enforce conservation only for those points 
      ! for which |erat|<maxrat 
      !-----------------------------------------------------------------

         if( ecfrc ) then
           if( efinl .ne. 0. ) then
             erat  = (efinl-ediff)/efinl
           endif
           if( abs(erat) .gt. maxrat .or. 
     $         abs(erat) .lt. (c1/maxrat) ) then
              erat = c1
           endif
           do nc = 1,ncat
             do layer = 1,nilay(nc)
               ein(layer,nc) = erat*ein(layer,nc)
             enddo
             vsn(nc) = erat*vsn(nc)
           enddo
           vsno(i,j) = c0
           do nc = 1,ncat
             vsno(i,j) = vsno(i,j)+vsn(nc)
           enddo
         endif 

      end subroutine conservation_check

c=======================================================================

      end module ice_therm_driver

c=======================================================================
