c $Id$
c=======================================================================
!---! parameters and commons blocks for MPI parallelization 
!---! internal to ice model
!---!
!---! authors Tony Craig, NCAR
!---!         Elizabeth C. Hunke, LANL
c=======================================================================

      module ice_mpi_internal

      use ice_kinds_mod
      use ice_domain

      implicit none

      integer (kind=int_kind), parameter ::
     &   nproc_x = 1    ! number of processors assigned in x direction
     &,  nproc_y = 1    ! number of processors assigned in y direction
     &,  nproc_s = nproc_x*nproc_y  ! total number of processors assigned, 
                                    ! both directions

      integer (kind=int_kind) ::
     &   nb_tasks               ! total number of tasks
     &,  ierr                   ! general-use error flag
     &,  MPI_COMM_ICE           ! communicator for internal ice comms







c=======================================================================

      contains

c=======================================================================

      subroutine ice_global_real_minmax(nc,work,string)

!---!-------------------------------------------------------------------
!---! determines and writes both minimum and maximum over global grid 
!---!
!---! author Tony Craig, NCAR
!---!-------------------------------------------------------------------

      integer (kind=int_kind), intent(in) :: nc
      real (kind=dbl_kind), intent(in) :: work(nc)
      character (len=8), intent(in) :: string
      real (kind=dbl_kind) :: amin,amax

      amin = ice_global_real_minval(nc,work)
      amax = ice_global_real_maxval(nc,work)
      if (my_task.eq.master_task) 
     &    write (6,*) ' min/max ',string,amin,amax

      end subroutine ice_global_real_minmax

c=======================================================================

      real function ice_global_real_minval(nc,work)

!---!-------------------------------------------------------------------
!---! computes minimum over the global grid
!---!
!---! author Tony Craig, NCAR
!---!-------------------------------------------------------------------

      integer (kind=int_kind), intent(in) :: nc
      integer (kind=int_kind) :: n
      real (kind=dbl_kind), intent(in) :: work(nc)
      real (kind=dbl_kind) :: local_val

      local_val = minval(work)

      ice_global_real_minval=local_val


      end function ice_global_real_minval

c=======================================================================

      real function ice_global_real_maxval(nc,work)

!---!-------------------------------------------------------------------
!---! computes maximum over the global grid
!---!
!---! author Tony Craig, NCAR
!---!-------------------------------------------------------------------

      integer (kind=int_kind), intent(in) :: nc
      integer (kind=int_kind) :: n
      real (kind=dbl_kind), intent(in) :: work(nc)
      real (kind=dbl_kind) :: local_val

      local_val = maxval(work)

      ice_global_real_maxval=local_val


      end function ice_global_real_maxval

c=======================================================================

      real function ice_global_real_sum(nc,work)

!---!-------------------------------------------------------------------
!---! sums given array over the global grid
!---!
!---! author Tony Craig, NCAR
!---!-------------------------------------------------------------------

      integer (kind=int_kind), intent(in) :: nc
      real (kind=dbl_kind), intent(in) :: work(nc)
      real (kind=dbl_kind) :: local_val

      local_val = sum(work)

      ice_global_real_sum=local_val


      end function ice_global_real_sum

c=======================================================================

      subroutine ice_bcast_rscalar(val)

!---!-------------------------------------------------------------------
!---! broadcasts real scalar value to all processors
!---!
!---! author Tony Craig, NCAR
!---!-------------------------------------------------------------------

      real (kind=dbl_kind), intent(in) :: val


      end subroutine ice_bcast_rscalar

c=======================================================================

      subroutine ice_bcast_iscalar(ival)

!---!-------------------------------------------------------------------
!---! broadcasts integer scalar value to all processors
!---!
!---! author Tony Craig, NCAR
!---!-------------------------------------------------------------------

      integer (kind=int_kind), intent(in) :: ival



      end subroutine ice_bcast_iscalar

c=======================================================================

      subroutine global_scatter(workg,work)

!---!-------------------------------------------------------------------
!---! scatters a global array and adds ghost cells to create a 
!---! distributed array
!---!
!---! author Tony Craig, NCAR
!---!-------------------------------------------------------------------

      use ice_model_size
      use ice_constants

      integer (kind=int_kind) :: i,j
      real (kind=dbl_kind) :: workg(imt_global,jmt_global)
     &,   work(ilo:ihi,jlo:jhi)


      ! shift indices from global domain to local domain
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
      do i=ilo,ihi
        work(i,j)=workg(i-num_ghost_cells,j-num_ghost_cells)
      enddo
      enddo


      end subroutine global_scatter

c=======================================================================

      subroutine global_gather(workg,work)

!---!-------------------------------------------------------------------
!---! gathers a distributed array and strips off ghost
!---! cells to create a local array with global dimensions
!---!
!---! author Tony Craig, NCAR
!---!-------------------------------------------------------------------

      use ice_model_size
      use ice_constants

      integer (kind=int_kind) :: i,j
      real (kind=dbl_kind) :: workg(imt_global,jmt_global)
     &,   work(ilo:ihi,jlo:jhi)


      ! shift indices from local domain to global domain
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
      do i=ilo,ihi
        workg(i-num_ghost_cells,j-num_ghost_cells)=work(i,j)
      enddo
      enddo


      end subroutine global_gather

c=======================================================================

      subroutine get_sum(flag,work1,work2,work3,gsum)

!---!-------------------------------------------------------------------
!---! computes a (weighted) sum over the global grid
!---! if flag = 1 then work1 is weighted by work2 before being
!---! added to work3
!---!
!---! author Elizabeth C. Hunke, LANL
!---!-------------------------------------------------------------------

      integer (kind=int_kind), intent(in) :: flag

      real (kind=dbl_kind), intent(in) ::
     &   work1(ilo:ihi,jlo:jhi)
     &,  work2(ilo:ihi,jlo:jhi)

      real (kind=dbl_kind), intent(in), dimension(:,:) ::
     &  work3

      real (kind=dbl_kind), intent(out) ::
     &   gsum

      real (kind=dbl_kind) ::
     &   work (ilo:ihi,jlo:jhi)

      integer (kind=int_kind) :: i,j
 
      do j=jlo,jhi
       do i=ilo,ihi
        if (flag.eq.1) then
          work(i,j) = work1(i,j) * work2(i,j)
        else
          work(i,j) = work1(i,j)
        endif
        work(i,j) = work(i,j) * work3(i,j)
       enddo
      enddo
      gsum = ice_global_real_sum((jhi-jlo+1)*(ihi-ilo+1),work)

      end subroutine get_sum

c=======================================================================

      subroutine end_run



      end subroutine end_run

c=======================================================================

      end module ice_mpi_internal

c=======================================================================
