c $Id$
c=======================================================================
!---! Elastic-viscous-plastic sea ice dynamics model 
!---! Computes ice velocity
!---!
!---! author Elizabeth C. Hunke
!---!        Fluid Dynamics Group, Los Alamos National Laboratory
!---!
!---! See E. C. Hunke and J. K. Dukowicz. An elastic-viscous-plastic 
!---!     model for sea ice dynamics. J. Phys. Oceanogr., 1997.
!---!
!---! Copyright, 1999.  The Regents of the University of California.
!---! This software was produced under a U.S. Government contract 
!---! (W-7405-ENG-36) by Los Alamos National Laboratory, which is 
!---! operated by the University of California for the U.S. Department 
!---! of Energy.  The U.S. Government is licensed to use, reproduce, and 
!---! distribute this software.  Permission is granted to the public to 
!---! copy and use this software without charge, provided that this 
!---! Notice and any statement of authorship are reproduced on all 
!---! copies.  Neither the Government nor the University makes any 
!---! warranty, express or implied, or assumes any liability or 
!---! responsibility for the use of this software.
c=======================================================================

      module ice_dyn_evp

      use ice_kinds_mod
      use ice_domain
      use ice_grid
      use ice_constants

      implicit none

      integer (kind=int_kind) ::
     &   kdyn         ! type of dynamics ( 1 = evp )
     &,  ndte         ! number of subcycles:  ndte=dt/dte

      logical (kind=log_kind) ::
     &   evp_damping  ! if true, use evp damping procedure

      real (kind=dbl_kind), parameter ::
     &   dragw = 0.00536_dbl_kind * rhow 
                      ! drag coefficient for water on ice *rhow (kg/m^3)
     &,  eyc = 0.36_dbl_kind
                      ! coefficient for calculating the parameter E
     &,  cosw = c1    ! cos(ocean turning angle)  ! turning angle = 0
     &,  sinw = c0    ! sin(ocean turning angle)  ! turning angle = 0

      real (kind=dbl_kind) ::
     &   u0           ! constant coefficient for initial u field (m/s)
     &,  v0           ! constant coefficient for initial v field (m/s)
     &,  ecci         ! 4/e^2 for the elliptical yield curve 
     &,  eccm         ! 2.-0.5*ecci
     &,  eccp         ! 1.+ecci*0.25
     &,  acoef        ! cryptic combinations of stress equation
     &,  ccoef        !   coefficients
     &,  bcoef        ! 
     &,  Tdtei        ! 2(wave damping timescale)/dte

      real (kind=dbl_kind), dimension (imt_local,jmt_local) ::
     &   u            ! x-component of velocity (m/s)
     &,  v            ! y-component of velocity (m/s)

      real (kind=dbl_kind), dimension (ilo:ihi,jlo:jhi) ::
     &   waterx       ! for ocean stress calculation, x (m/s)
     &,  watery       ! for ocean stress calculation, y (m/s)
     &,  forcex       ! work array: combined atm stress and ocn tilt, x
     &,  forcey       ! work array: combined atm stress and ocn tilt, y
     &,  umassdtei    ! mass of U-cell/dte (kg/m^2 s)
     &,  fcor         ! Coriolis parameter (1/s)
     &,  fm           ! Coriolis param. * mass in U-cell (kg/s)
     &,  rcon         ! for damping criterion (kg/s)
     &,  prss         ! pressure P (centered in T-cell) (kg/s)
     &,  sig11ne      ! ice stress tensor: sigma_11 north
     &,  sig11sw      !   sigma_11, east       (kg/s^2)
     &,  sig11nw      !   sigma_11, south
     &,  sig11se      !   sigma_11, west
     &,  sig12ne      !   sigma_12, north
     &,  sig12sw      !   sigma_12, east
     &,  sig12nw      !   sigma_12, south
     &,  sig12se      !   sigma_12, west
     &,  sig22ne      !   sigma_22, north
     &,  sig22sw      !   sigma_22, east
     &,  sig22nw      !   sigma_22, south
     &,  sig22se      !   sigma_22, west
     &,  prs_sig      ! replacement pressure, for stress calc
     &,  strintx        ! divergence of internal ice stress, x (N/m^2)
     &,  strinty        ! divergence of internal ice stress, y (N/m^2)
     &,  sig1         ! principal stress component (diagnostic)
     &,  sig2         ! principal stress component (diagnostic)

c=======================================================================

      contains

c=======================================================================

      subroutine evp

!---!-------------------------------------------------------------------
!---! elastic-viscous-plastic dynamics driver
!---!-------------------------------------------------------------------

      use ice_timers

      integer (kind=int_kind) :: k

      call ice_timer_start(2)  ! dynamics

      call evp_prep            ! preparation for dynamics

      do k=1,ndte              ! subcycling
        call stress            ! the stress tensor equation
        call stepu             ! total surface stress, momentum equation
        call bound_sw(u)       ! Periodic/Neumann boundary conditions
        call bound_sw(v)       ! Periodic/Neumann boundary conditions
      enddo

      call evp_finish          ! ice-ocean stress

      call ice_timer_stop(2)   ! dynamics

      end subroutine evp

c=======================================================================

      subroutine init_evp

!---!-------------------------------------------------------------------
!---! Initialize parameters and variables needed for the evp dynamics
!---!-------------------------------------------------------------------

      use ice_calendar

      integer (kind=int_kind) :: i, j
      real (kind=dbl_kind) ::  
     &   dte                   ! subcycling timestep for EVP dynamics, s
     &,  ecc, c2n, ecoef, d2n, dcoef, tdamp2

      ! elastic time step
      dte = dt/real(ndte)        ! s
      dtei = c1/dte              ! 1/s
      tdamp2 = c2*eyc*dt         ! 2(wave damping time scale T)
      if (my_task.eq.master_task) 
     &  write(6,*) 'dt = ',dt,'  dte = ',dte,'  tdamp = ',p5*tdamp2
      ! major/minor axis length ratio, squared
      ecc  = c4
      ecci = c1                  ! 4./ecc
      eccm = c2-p5*ecci
      eccp = c1+ecci*p25
      ! constants for stress equation
      Tdtei = dtei*tdamp2
      c2n   = Tdtei + ecc
      ecoef = p5*(ecc-c1)
      d2n   = c2n - ecoef
      ccoef = c1/c2n
      dcoef = ccoef/(Tdtei + c1)
      bcoef = dcoef*d2n
      acoef = dcoef*ecoef

c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
        ! coefficient for damping criterion
        rcon(i,j) = 615.0_dbl_kind*eyc*dt*(dtei*HTN(i,j))**2  ! kg/s

        ! Coriolis parameter
c        fcor(i,j) = 1.46e-4_dbl_kind ! Hibler 1979, Northern Hemisphere; 1/s
        fcor(i,j) = c2*(7.292e-5_dbl_kind)*sin(ULAT(i,j))  ! 1/s

        ! velocity
        if (umask(i,j)) then
         u(i,j) = c0       ! m/s
         v(i,j) = c0       ! m/s
        endif

        ! stress tensor,  kg/s^2
        ! 3 components (sigma_11, sigma_12=sigma_21, sigma_22)
        ! 4 overlapping triangles per grid cell (northeast, southeast, etc)
        sig11ne(i,j) = c0
        sig11se(i,j) = c0
        sig11sw(i,j) = c0
        sig11nw(i,j) = c0

        sig22ne(i,j) = c0
        sig22se(i,j) = c0
        sig22sw(i,j) = c0
        sig22nw(i,j) = c0

        sig12ne(i,j) = c0
        sig12se(i,j) = c0
        sig12sw(i,j) = c0
        sig12nw(i,j) = c0
       enddo
      enddo

      call bound(u)
      call bound(v)

      end subroutine init_evp

c=======================================================================

      subroutine evp_prep

!---!-------------------------------------------------------------------
!---! Computes quantities needed in the stress tensor (sigma) 
!---! and momentum (u) equations, but which do not change during 
!---! the thermodynamics/transport time step:
!---!   wind stress shift to U grid,
!---!   ice mass and ice extent masks,
!---!   pressure (strength), and part of the forcing stresses
!---! initializes ice velocity for new points to ocean sfc current
!---!-------------------------------------------------------------------

      use ice_flux
      use ice_calendar
      use ice_state

      integer (kind=int_kind) :: i, j, n
      real (kind=dbl_kind) :: 
     &   umass(ilo:ihi,jlo:jhi)
     &,  aiu(ilo:ihi,jlo:jhi)   ! ice fraction on u-grid
      logical (kind=log_kind) :: tmphm(imt_local,jmt_local)
      logical (kind=log_kind) :: iceumask_old(ilo:ihi,jlo:jhi)

      !-----------------------------------------------------------------
      ! total mass of ice and snow, centered in T-cell
      !-----------------------------------------------------------------
c$OMP PARALLEL DO PRIVATE(i,j,n)
      do j=1,jmt_local
       do i=1,imt_local
        if (tmask(i,j)) then
         tmass(i,j) = (rhoi*vice(i,j) + rhos*vsno(i,j)) ! kg/m^2
        endif
       enddo
      enddo

      !-----------------------------------------------------------------
      ! convert dynamics variables to U grid
      !-----------------------------------------------------------------
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
      strairx(i,j) = strairxT(i,j) ! prep to convert to U grid
      strairy(i,j) = strairyT(i,j)
       enddo
      enddo
      call t2ugrid(strairx)
      call t2ugrid(strairy)
      call to_ugrid(tmass,umass)
      call to_ugrid(aice,aiu)

      !-----------------------------------------------------------------
      ! convenient variable for evp
      !-----------------------------------------------------------------
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
        umassdtei(i,j) = umass(i,j)*dtei               ! m/dte, kg/m^2 s
       enddo
      enddo

      !-----------------------------------------------------------------
      ! augmented masks (land + open ocean)
      !-----------------------------------------------------------------
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=1,jmt_local
       do i=1,imt_local
        ! ice extent mask (T-cells)
        tmphm(i,j) = tmask(i,j) .and. (aice(i,j).gt.puny)
       enddo
      enddo
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
        ! extend ice extent mask (T-cells) to points around pack
        icetmask(i,j) =  
     &   tmphm(i-1,j+1) .or. tmphm(i,j+1) .or. tmphm(i+1,j+1) .or.
     &   tmphm(i-1,j)   .or. tmphm(i,j)   .or. tmphm(i+1,j)   .or.
     &   tmphm(i-1,j-1) .or. tmphm(i,j-1) .or. tmphm(i+1,j-1)
        icetmask(i,j) = icetmask(i,j) .and. tmask(i,j) ! remask land points
        ! ice extent mask (U-cells)
         iceumask_old(i,j) = iceumask(i,j)  ! save
        iceumask(i,j) = (umask(i,j)) .and. (aiu(i,j).gt.puny)   
       enddo
      enddo

      !-----------------------------------------------------------------
      ! pressure and forcing terms; set sigma=0 for no ice;
      ! initialize ice velocity in cells previously empty to ocn current
      !-----------------------------------------------------------------
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
       do i=ilo,ihi
       if (icetmask(i,j)) then
        prss(i,j) = strength(i,j)
        fm(i,j) = fcor(i,j)*umass(i,j)               ! Coriolis * mass

        if (umask(i,j)) then
        ! for ocean stress
        waterx(i,j) = uocn(i,j)*cosw - vocn(i,j)*sinw
        watery(i,j) = vocn(i,j)*cosw + uocn(i,j)*sinw

        ! calculate tilt from geostrophic currents if needed
c        strtltx(i,j) = -fm(i,j)*vocn(i,j)
c        strtlty(i,j) =  fm(i,j)*uocn(i,j)
        ! combine tilt with wind stress
        forcex(i,j) = strairx(i,j) - fm(i,j)*vocn(i,j)
        forcey(i,j) = strairy(i,j) + fm(i,j)*uocn(i,j)

        endif  ! umask
       endif  ! icetmask

       if (.not.tmphm(i,j)) then
       sig11ne(i,j) = c0
       sig11sw(i,j) = c0
       sig11nw(i,j) = c0
       sig11se(i,j) = c0

       sig22ne(i,j) = c0
       sig22sw(i,j) = c0
       sig22nw(i,j) = c0
       sig22se(i,j) = c0

       sig12ne(i,j) = c0
       sig12sw(i,j) = c0
       sig12nw(i,j) = c0
       sig12se(i,j) = c0
       endif

        ! initialize velocity for new ice points to ocean sfc current
        if( iceumask(i,j) .and. (.not. iceumask_old(i,j))) then
          u(i,j) = uocn(i,j)
          v(i,j) = vocn(i,j)
        endif 

       enddo
      enddo

      end subroutine evp_prep

c=======================================================================

      subroutine stress

!---!-------------------------------------------------------------------
!---! Computes the rates of strain and internal stress components for 
!---! each of the four triangles in each grid cell (north, south, east, 
!---! west).
!---!-------------------------------------------------------------------

      integer (kind=int_kind) :: i, j
      real (kind=dbl_kind) :: 
     & dun,due,dus,duw,      ! du for each cell edge (n,e,s,w)
     & dvn,dve,dvs,dvw,      ! dv for each cell edge (n,e,s,w)
     & xi11ne,xi12ne,xi22ne, ! strain rates for ne triangle
     & xi11sw,xi12sw,xi22sw, ! strain rates for sw triangle
     & xi11nw,xi12nw,xi22nw, ! strain rates for nw triangle
     & xi11se,xi12se,xi22se, ! strain rates for se triangle
     & deltane,deltase,deltasw,deltanw, ! Delta for each triangle
     & c4ne,c4sw,c4nw,c4se,c5ne,c5sw,c5nw,c5se, ! cryptic stuff
     & prsdne, prsdsw, prsdnw, prsdse,  ! replacement pressure
     & prs2ne, prs2sw, prs2nw, prs2se   ! 0.5 * pressure

c$OMP PARALLEL DO PRIVATE(i,j,
c$OMP&   dun,due,dus,duw,dvn,dve,dvs,dvw,
c$OMP&   xi11ne,xi12ne,xi22ne,xi11sw,xi12sw,xi22sw,xi11nw,xi12nw,xi22nw,
c$OMP&   xi11se,xi12se,xi22se,c4ne,c4sw,c4nw,c4se,c5ne,c5sw,c5nw,c5se,
c$OMP&   deltane,deltasw,deltanw,deltase,
c$OMP&   prsdne,prsdsw,prsdnw,prsdse,prs2ne,prs2sw,prs2nw,prs2se)
      do j=jlo,jhi
       do i=ilo,ihi

      if (icetmask(i,j)) then

      dun = (u(i,j) - u(i-1,j))*dxtr(i,j)
      dus = (u(i,j-1) - u(i-1,j-1))*dxtr(i,j)
      due = (u(i,j) - u(i,j-1))*dytr(i,j)
      duw = (u(i-1,j) - u(i-1,j-1))*dytr(i,j)

      dvn = (v(i,j) - v(i-1,j))*dxtr(i,j)
      dvs = (v(i,j-1) - v(i-1,j-1))*dxtr(i,j)
      dve = (v(i,j) - v(i,j-1))*dytr(i,j)
      dvw = (v(i-1,j) - v(i-1,j-1))*dytr(i,j)

      !-----------------------------------------------------------------
      ! rates of strain                                ! 1/s
      !-----------------------------------------------------------------
      xi11ne = dun
      xi11sw = dus
      xi11nw = dun
      xi11se = dus

      xi12ne = p5*(due + dvn)
      xi12sw = p5*(duw + dvs)
      xi12nw = p5*(duw + dvn)
      xi12se = p5*(due + dvs)

      xi22ne = dve
      xi22sw = dvw
      xi22nw = dvw
      xi22se = dve

      !-----------------------------------------------------------------
      ! Delta (in the denominator of zeta, eta)        ! 1/s
      !-----------------------------------------------------------------
      deltane = sqrt( (xi11ne**2+xi22ne**2)*eccp
     &  + xi12ne**2*ecci + xi11ne*xi22ne*eccm)
      deltasw = sqrt( (xi11sw**2+xi22sw**2)*eccp
     &  + xi12sw**2*ecci + xi11sw*xi22sw*eccm)
      deltanw = sqrt( (xi11nw**2+xi22nw**2)*eccp
     &  + xi12nw**2*ecci + xi11nw*xi22nw*eccm)
      deltase = sqrt( (xi11se**2+xi22se**2)*eccp
     &  + xi12se**2*ecci + xi11se*xi22se*eccm)

      !-----------------------------------------------------------------
      ! replacement pressure P/Delta                   ! kg/s
      !-----------------------------------------------------------------
      if (evp_damping) then
        ! enforce damping criterion
        prsdne = min(prss(i,j)/max(eps11,deltane),rcon(i,j))
        prsdsw = min(prss(i,j)/max(eps11,deltasw),rcon(i,j))
        prsdnw = min(prss(i,j)/max(eps11,deltanw),rcon(i,j))
        prsdse = min(prss(i,j)/max(eps11,deltase),rcon(i,j))

      else
        ! original version
        prsdne = prss(i,j)/max(eps11,deltane)
        prsdsw = prss(i,j)/max(eps11,deltasw)
        prsdnw = prss(i,j)/max(eps11,deltanw)
        prsdse = prss(i,j)/max(eps11,deltase)
      endif

      ! P/2 
      prs2ne = p5*prsdne*deltane
      prs2sw = p5*prsdsw*deltasw
      prs2nw = p5*prsdnw*deltanw
      prs2se = p5*prsdse*deltase

      ! save replacement pressure for principal stress calculation
      if (evp_damping) then   ! match triangle with that used 
                              ! in routine principal_stress
        prs_sig(i,j) =  prss(i,j)*deltane/max(eps11,deltane)
      else
        prs_sig(i,j) = c2*prs2ne 
      endif

      !-----------------------------------------------------------------
      ! cryptic stuff
      !-----------------------------------------------------------------
      c4ne =  Tdtei*sig11ne(i,j) + prsdne*xi11ne - prs2ne
      c4sw =  Tdtei*sig11sw(i,j) + prsdsw*xi11sw - prs2sw
      c4nw =  Tdtei*sig11nw(i,j) + prsdnw*xi11nw - prs2nw
      c4se =  Tdtei*sig11se(i,j) + prsdse*xi11se - prs2se

      c5ne =  Tdtei*sig22ne(i,j) + prsdne*xi22ne - prs2ne
      c5sw =  Tdtei*sig22sw(i,j) + prsdsw*xi22sw - prs2sw
      c5nw =  Tdtei*sig22nw(i,j) + prsdnw*xi22nw - prs2nw
      c5se =  Tdtei*sig22se(i,j) + prsdse*xi22se - prs2se

      !-----------------------------------------------------------------
      ! finally, the stresses                            ! kg/s^2
      !-----------------------------------------------------------------
      sig11ne(i,j) = acoef*c5ne + c4ne*bcoef
      sig11sw(i,j) = acoef*c5sw + c4sw*bcoef
      sig11nw(i,j) = acoef*c5nw + c4nw*bcoef
      sig11se(i,j) = acoef*c5se + c4se*bcoef

      sig22ne(i,j) = acoef*c4ne + c5ne*bcoef
      sig22sw(i,j) = acoef*c4sw + c5sw*bcoef
      sig22nw(i,j) = acoef*c4nw + c5nw*bcoef
      sig22se(i,j) = acoef*c4se + c5se*bcoef

      sig12ne(i,j) = (Tdtei*sig12ne(i,j) + prsdne*xi12ne)*ccoef
      sig12sw(i,j) = (Tdtei*sig12sw(i,j) + prsdsw*xi12sw)*ccoef
      sig12nw(i,j) = (Tdtei*sig12nw(i,j) + prsdnw*xi12nw)*ccoef
      sig12se(i,j) = (Tdtei*sig12se(i,j) + prsdse*xi12se)*ccoef

      endif

       enddo
      enddo

      end subroutine stress

c=======================================================================

      subroutine stepu

!---!-------------------------------------------------------------------
!---! Calculation of the surface stresses
!---! Integration of the momentum equation to find velocity (u,v)
!---!-------------------------------------------------------------------

      use ice_flux

      integer (kind=int_kind) :: i, j
      real (kind=dbl_kind) :: 
     &   vrel,cca,ccb,ab2,s11,s12,s21,s22,cc1,cc2, taux,tauy
     &,  s1122(8,imt_local,jmt_local),work(imt_local,jmt_local)

      !-----------------------------------------------------------------
      ! a cryptic but useful array
      !-----------------------------------------------------------------
      ! s1122(1)=s11n, s1122(2)=s11s, s1122(3)=s22e, s1122(4)=s22w
      ! s1122(5)=s12n, s1122(6)=s12s, s1122(7)=s12e, s1122(8)=s12w
      !-----------------------------------------------------------------
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
      do i=ilo,ihi
      if (icetmask(i,j)) then
       s1122(1,i,j) = (sig11nw(i,j) + sig11ne(i,j))*dxtr4(i,j)
       s1122(2,i,j) = (sig11sw(i,j) + sig11se(i,j))*dxtr4(i,j)
       s1122(3,i,j) = (sig22se(i,j) + sig22ne(i,j))*dytr4(i,j)
       s1122(4,i,j) = (sig22sw(i,j) + sig22nw(i,j))*dytr4(i,j)
       s1122(5,i,j) = (sig12nw(i,j) + sig12ne(i,j))*dxtr4(i,j)
       s1122(6,i,j) = (sig12sw(i,j) + sig12se(i,j))*dxtr4(i,j)
       s1122(7,i,j) = (sig12se(i,j) + sig12ne(i,j))*dytr4(i,j)
       s1122(8,i,j) = (sig12sw(i,j) + sig12nw(i,j))*dytr4(i,j)
      else
       s1122(1,i,j) = c0
       s1122(2,i,j) = c0
       s1122(3,i,j) = c0
       s1122(4,i,j) = c0
       s1122(5,i,j) = c0
       s1122(6,i,j) = c0
       s1122(7,i,j) = c0
       s1122(8,i,j) = c0
      endif
      enddo
      enddo
      call bound_narr_ne(8,s1122)

      !-----------------------------------------------------------------
      ! integrate the momentum equation
      !-----------------------------------------------------------------
c$OMP PARALLEL DO PRIVATE(i,j,vrel,cca,ccb,ab2,
c$OMP&   s11,s12,s21,s22,cc1,cc2,taux,tauy)
      do j=jlo,jhi
      do i=ilo,ihi

       if (iceumask(i,j)) then
        ! (magnitude of relative ocean current)*rhow*drag
        vrel = dragw*sqrt((uocn(i,j) - u(i,j))**2 + 
     &                    (vocn(i,j) - v(i,j))**2)  ! m/s
        ! ice/ocean stress
        taux = vrel*waterx(i,j) ! NOTE this is not the entire
        tauy = vrel*watery(i,j) ! ocn stress term

        ! alpha, beta are defined in Hunke and Dukowicz (1997), section 3.2
        cca = umassdtei(i,j) + vrel         ! alpha, kg/m^2 s
        ccb = fm(i,j)                       ! beta,  kg/m^2 s
        ab2 = cca**2 + ccb**2

        ! more cryptic stuff
        s11=s1122(2,i+1,j+1)-s1122(2,i,j+1)+s1122(1,i+1,j)-s1122(1,i,j)
        s22=s1122(4,i+1,j+1)-s1122(4,i+1,j)+s1122(3,i,j+1)-s1122(3,i,j)
        s21=s1122(6,i+1,j+1)-s1122(6,i,j+1)+s1122(5,i+1,j)-s1122(5,i,j)
        s12=s1122(8,i+1,j+1)-s1122(8,i+1,j)+s1122(7,i,j+1)-s1122(7,i,j)

        ! finally, the velocity components
        strintx(i,j) = s11 + s12
        strinty(i,j) = s21 + s22
        cc1 = strintx(i,j) + forcex(i,j) + taux + umassdtei(i,j)*u(i,j)
        cc2 = strinty(i,j) + forcey(i,j) + tauy + umassdtei(i,j)*v(i,j)

        u(i,j) = (cca*cc1 + ccb*cc2)/ab2              ! m/s
        v(i,j) = (cca*cc2 - ccb*cc1)/ab2

      !-----------------------------------------------------------------
      ! ocean-ice stress for coupling 
      !-----------------------------------------------------------------
        strocnx(i,j) = taux  
        strocny(i,j) = tauy

      else
      !-----------------------------------------------------------------
      ! set velocity and stress to zero on land and (nearly) open water
      !-----------------------------------------------------------------
        u(i,j) = c0
        v(i,j) = c0
        strocnx(i,j) = c0
        strocny(i,j) = c0
        strintx(i,j) = c0
        strinty(i,j) = c0
      endif

      enddo
      enddo

      end subroutine stepu

c=======================================================================

      subroutine evp_finish

!---!-------------------------------------------------------------------
!---! calculation of the ice-ocean stress
!---! ...the sign will be reversed later...
!---!-------------------------------------------------------------------

      use ice_flux

      real (kind=dbl_kind) :: vrel
      integer (kind=int_kind) :: i, j

c$OMP PARALLEL DO PRIVATE(i,j,vrel,factor,UTE,UTW,VTN,VTS)
      do j=jlo,jhi
      do i=ilo,ihi
      if (iceumask(i,j)) then
        ! ocean-ice stress for coupling
        vrel = dragw*sqrt((uocn(i,j) - u(i,j))**2 + 
     &                    (vocn(i,j) - v(i,j))**2)  ! m/s
        strocnx(i,j) = strocnx(i,j) - vrel*(u(i,j)*cosw - v(i,j)*sinw)
        strocny(i,j) = strocny(i,j) - vrel*(v(i,j)*cosw + u(i,j)*sinw)
      else
        ! set stress to zero on land and (nearly) open water
        strocnx(i,j) = c0
        strocny(i,j) = c0
      endif

      !-----------------------------------------------------------------
      ! convert strocn to T grid
      !-----------------------------------------------------------------
      strocnxT(i,j) = strocnx(i,j)  ! prepare to shift
      strocnyT(i,j) = strocny(i,j)

      enddo
      enddo

      call u2tgrid(strocnxT)        ! shift
      call u2tgrid(strocnyT)

      end subroutine evp_finish

c=======================================================================

      subroutine principal_stress

!---!-------------------------------------------------------------------
!---! computes principal stresses for comparison with the theoretical 
!---! yield curve
!---!-------------------------------------------------------------------

      real (kind=dbl_kind) :: 
     &    sig11(ilo:ihi,jlo:jhi),sig12(ilo:ihi,jlo:jhi)
     &,   sig22(ilo:ihi,jlo:jhi),diff
      integer (kind=int_kind) :: i, j

      !-----------------------------------------------------------------
      ! choose one set of stresses
      !-----------------------------------------------------------------
      ! averaging throws stresses off the yield curve, especially at 
      ! the corners
      !-----------------------------------------------------------------
c$OMP PARALLEL DO PRIVATE(i,j)
      do j=jlo,jhi
        do i=ilo,ihi
          sig11(i,j) = sig11ne(i,j)
          sig12(i,j) = sig12ne(i,j)
          sig22(i,j) = sig22ne(i,j)
        enddo
      enddo

      !-----------------------------------------------------------------
      ! principal stresses
      !-----------------------------------------------------------------
c$OMP PARALLEL DO PRIVATE(i,j,diff)
      do j=jlo,jhi
        do i=ilo,ihi
          diff=sig11(i,j)-sig22(i,j)
          if(prs_sig(i,j).gt.eps11) then
            sig1(i,j)=(p5*(sig11(i,j)+sig22(i,j)
     &              +sqrt(diff**2+c4*sig12(i,j)**2)))/prs_sig(i,j)
            sig2(i,j)=(p5*(sig11(i,j)+sig22(i,j)
     &              -sqrt(diff**2+c4*sig12(i,j)**2)))/prs_sig(i,j)
          else 
            sig1(i,j)=c1000
            sig2(i,j)=c1000
          endif
        enddo
      enddo

      end subroutine principal_stress

c=======================================================================

      end module ice_dyn_evp

c=======================================================================
