c $Id: ice_diagnostics.F,v 1.2 2001/02/23 21:06:52 schramm Exp $
c=======================================================================
!---! diagnostic information output during run
!---!
!---! authors  (?)
!---!         Elizabeth C. Hunke, LANL
c=======================================================================

      module ice_diagnostics

      use ice_domain
      use ice_constants
      use sys_column

      implicit none

      ! for water and heat budgets
      real (kind=dbl_kind) ::
     &   totmn   ! total ice/snow water mass (nh)
     &,  totms   ! total ice/snow water mass (sh)
     &,  toten   ! total ice/snow enthalpy (J)
     &,  totes   ! total ice/snow enthalpy (J)

      ! point print data
      logical (kind=log_kind) ::
     &    print_points             ! if true, print point data
      integer (kind=int_kind), parameter ::
     &          npnt = 12  ! total number of points to be printed

      integer (kind=int_kind) ::
     $         iunit   ! temporary for print

      real (kind=dbl_kind), dimension(npnt) :: 
     &     latpnt,  !  latitude of desired points
     $     lonpnt   ! longitude of desired points
      integer (kind=int_kind) ::
     &     untout(npnt)  ! unit of output points
     &,     iindx(npnt)  ! i index for points
     &,     jindx(npnt)  ! j index for points

      data latpnt /   80.,  86.,   60.,   58.,   70.,  45.,
     $               -65., -65.,  -74.,  -55.,  -60., -45./
      data lonpnt /  165., 180.,  185.,  280.,  345., 330.,
     $               180., 320.,  190.,   90.,   30., 330./
      data untout /    91,    92,    93,    94,   95,  96,
     $                 81,    82,    83,    84,   85,  86/

      ! printing info for routine print_state
      character (len=20) :: plabel
      integer (kind=int_kind), parameter ::
     &  check_step = 100000
     &, ip = 51
     &, jp = 37
     &, mtask = 2

c=======================================================================

      contains

c=======================================================================

      subroutine runtime_diags

!---!-------------------------------------------------------------------
!---! Writes diagnostic info (max, min, global sums, etc) to standard out
!---!
!---! authors Elizabeth C. Hunke, LANL
!---!          (?)
!---!-------------------------------------------------------------------

      use ice_model_size
      use ice_flux
      use ice_albedo
      use ice_mpi_internal
      use ice_history
      use ice_grid
      use ice_calendar
      use ice_state
      use ice_dyn_evp
      use ice_mechred

      real (kind=dbl_kind) ::
     &   umaxn, hmaxn, shmaxn, arean, snwmxn, rnn, snn, frzn
     &,  umaxs, hmaxs, shmaxs, areas, snwmxs, rns, sns, frzs
     &,  etotn, hnetn, fhocnn, fhatmn, fhfrzn, atotn
     &,  etots, hnets, fhocns, fhatms, fhfrzs, atots
     &,  sfreshn, evpn, mtotn, micen, msnwn, fluxn
     &,  sfreshs, evps, mtots, mices, msnws, fluxs
     &,  delmin, delein, werrn, herrn
     &,  delmis, deleis, werrs, herrs
     &,  workn(imt_local,jmt_local), works(imt_local,jmt_local)
     &,  work1(ilo:ihi,jlo:jhi), work2(ilo:ihi,jlo:jhi), ftmp
      integer (kind=int_kind) :: n, layer, i,j
      logical (kind=log_kind) :: prwtbd   ! if true, print water budget terms
c      prwtbd = .false.
      prwtbd = .true.

      !-----------------------------------------------------------------
      ! state of the ice
      !-----------------------------------------------------------------

      ! maximum velocity  
      do j=1,jmt_local
       do i=1,imt_local
       workn(i,j) = max(abs(mask_n(i,j)*u(i,j)),abs(mask_n(i,j)*v(i,j)))
       works(i,j) = max(abs(mask_s(i,j)*u(i,j)),abs(mask_s(i,j)*v(i,j)))
       enddo
      enddo
      umaxn = ice_global_real_maxval(imt_local*jmt_local,WORKN)
      umaxs = ice_global_real_maxval(imt_local*jmt_local,WORKS)
      ! maximum effective thickness
      do j=1,jmt_local
       do i=1,imt_local
       workn(i,j) = mask_n(i,j)*vice(i,j)
       works(i,j) = mask_s(i,j)*vice(i,j)
       enddo
      enddo
      hmaxn = ice_global_real_maxval(imt_local*jmt_local,WORKN)
      hmaxs = ice_global_real_maxval(imt_local*jmt_local,WORKS)
      ! total ice area
      call get_sum(0,tarean,one,aice,arean)
      call get_sum(0,tareas,one,aice,areas)
      arean = arean * m2_to_km2
      areas = areas * m2_to_km2
      ! total ice volume
      call get_sum(0,tarean,one,vice,shmaxn)
      call get_sum(0,tareas,one,vice,shmaxs)
      ! total snow volume
      call get_sum(0,tarean,one,vsno,snwmxn)
      call get_sum(0,tareas,one,vsno,snwmxs)
      ! total ice/snow enthalpy
       do j=1,jmt_local
        do i=1,imt_local
         works(i,j) = c0
        enddo
       enddo
      do layer = 1,ntilay
       do j=1,jmt_local
        do i=1,imt_local
         works(i,j) = works(i,j) + eicen(i,j,layer) ! ice enthalpy
        enddo
       enddo
      enddo
      call get_sum(0,tarean,one,works,etotn)
      call get_sum(0,tareas,one,works,etots)
      etotn = etotn - snwmxn * rLfs   ! plus snow enthalpy
      etots = etots - snwmxs * rLfs 
      ! average ice albedo
       do j=jlo,jhi
        do i=ilo,ihi
         work1(i,j) = alvdr(i,j)*awtvdr + alidr(i,j)*awtidr 
     $              + alvdf(i,j)*awtvdf + alidf(i,j)*awtidf
        enddo
       enddo
      call get_sum(1,tarean,work1,aice,atotn)
      call get_sum(1,tareas,work1,aice,atots)
      atotn = atotn / arean * m2_to_km2  ! convert arean back to m2
      atots = atots / areas * m2_to_km2


      !-----------------------------------------------------------------
      ! various fluxes
      !-----------------------------------------------------------------
      ! evap, Fresh, Fhnet, Fswthr, Fsensible, Flwout 
      ! need to be multiplied by aice because 
      ! regretfully they have been divided by aice for the coupler
      !-----------------------------------------------------------------

      ! evaporation
      call get_sum(1,tarean,evap,aice_init,evpn)	
      call get_sum(1,tareas,evap,aice_init,evps)	
      evpn = evpn*dt
      evps = evps*dt
      ! fresh water flux
      call get_sum(1,tarean,Frsh_hist,aice_init,sfreshn) 
      call get_sum(1,tareas,Frsh_hist,aice_init,sfreshs) 
      sfreshn = sfreshn*dt
      sfreshs = sfreshs*dt
      ! ocean heat

      call get_sum(1,tarean,Foht_hist+Fswthru_hist,aice_init,fhocnn)
      call get_sum(1,tareas,Foht_hist+Fswthru_hist,aice_init,fhocns)

      ! latent heat
      ! you may be wondering, where is the latent heat flux? 
      ! I do not include it here because it cancels with 
      ! the evaporative flux times the enthalpy of the  
      ! ice/snow that evaporated

      ! atmo heat flux
      ! Note: Flwout includes the reflected longwave down, needed by the
      ! atmosphere as an upwards radiative boundary condition.
      do j=jlo,jhi
       do i=ilo,ihi
        ftmp = Fhdiag(i,j) + (Fsensible(i,j) + Flwout(i,j))*
     &		aice_init(i,j)
        work1(i,j) = tarean(i,j)*ftmp
        work2(i,j) = tareas(i,j)*ftmp
       enddo
      enddo
      fhatmn = ice_global_real_sum((jhi-jlo+1)*(ihi-ilo+1),WORK1)
      fhatms = ice_global_real_sum((jhi-jlo+1)*(ihi-ilo+1),WORK2)
      ! freezing potential
      do j=jlo,jhi
       do i=ilo,ihi
        ftmp = max(c0,frzmlt(i,j))
        work1(i,j) = tarean(i,j)*ftmp
        work2(i,j) = tareas(i,j)*ftmp
       enddo
      enddo
      fhfrzn = ice_global_real_sum((jhi-jlo+1)*(ihi-ilo+1),WORK1)
      fhfrzs = ice_global_real_sum((jhi-jlo+1)*(ihi-ilo+1),WORK2)
      ! rain
      call get_sum(1,tarean,Frain,aice_init,rnn)	
      call get_sum(1,tareas,Frain,aice_init,rns)	
      rnn = rnn*dt
      rns = rns*dt
      ! snow
      call get_sum(1,tarean,Fsnow,aice_init,snn)	
      call get_sum(1,tareas,Fsnow,aice_init,sns)	
      snn = snn*dt
      sns = sns*dt
      ! frazil ice growth !! should not be multiplied by aice
      do j=jlo,jhi
       do i=ilo,ihi
        work1(i,j) = tarean(i,j)*frazil(i,j)
        work2(i,j) = tareas(i,j)*frazil(i,j)
       enddo
      enddo
      frzn = ice_global_real_sum((jhi-jlo+1)*(ihi-ilo+1),WORK1)
      frzs = ice_global_real_sum((jhi-jlo+1)*(ihi-ilo+1),WORK2)
      frzn = frzn*dt
      frzs = frzs*dt

      ! ice and snow mass
      micen = rhoi*shmaxn
      msnwn = rhos*snwmxn
      mices = rhoi*shmaxs
      msnws = rhos*snwmxs

      mtotn = micen + msnwn
      mtots = mices + msnws

      delmin = mtotn - totmn - mass2col
      delmis = mtots - totms

      ! total water flux
      fluxn  = c0
      fluxs  = c0
      if( arean .gt. 0. ) then
        fluxn = rnn + snn + evpn - sfreshn + frzn
      endif
      if( areas .gt. 0. ) then
        fluxs = rns + sns + evps - sfreshs + frzs
      endif
      
      werrn = (fluxn-delmin)/(mtotn+c1)
      werrs = (fluxs-delmis)/(mtots+c1)

      ! enthalpy change
      delein = etotn - toten - enth2col
      deleis = etots - totes

      fhatmn = fhatmn + ( - snn * Lfus + evpn * Lvap ) / dt 
      fhatms = fhatms + ( - sns * Lfus + evps * Lvap ) / dt 

      hnetn = (fhatmn - fhocnn - fhfrzn) * dt
      hnets = (fhatms - fhocns - fhfrzs) * dt

      herrn = (hnetn - delein) / (etotn - c1)
      herrs = (hnets - deleis) / (etots - c1)

      !-----------------------------------------------------------------
      ! start spewing
      !-----------------------------------------------------------------
      if (my_task.eq.master_task) then

        write (6,899) 'Gridcell'

        write (6,900) 'max cH (m)             = ',hmaxn
        write (6,901) 'total area (km^2)      = ',arean
        write (6,901) 'total ice volume (m^3) = ',shmaxn
        write (6,901) 'total snw volume (m^3) = ',snwmxn
        write (6,900) 'average albedo         = ',atotn

        call shr_sys_flush(6)

        if( prwtbd ) then  
  
          write (6,903) 'arwt rain h2o kg in dt = ',rnn,' rnn' 
          write (6,903) 'arwt snow h2o kg in dt = ',snn,' snn' 
          write (6,903) 'arwt evap h2o kg in dt = ',evpn,' evpn' 
          write (6,903) 'arwt frzl h2o kg in dt = ',frzn,' frzn' 
          write (6,903) 'arwt frsh h2o kg in dt = ',sfreshn,' sfreshn' 

          write (6,903) 'arwt ice mass (kg)     = ',micen,' micen' 
          write (6,903) 'arwt snw mass (kg)     = ',msnwn,' msnwn' 

          write (6,903) 'arwt tot mass (kg)     = ',mtotn,' mtotn' 
          write (6,903) 'arwt tot mass chng(kg) = ',delmin,
     $         ' delmin' 
          write (6,903) 'arwt water flux        = ',fluxn,' fluxn' 
          write (6,*) '(=rain+snow+evap+frzl-frsh)  '
          write (6,904) 'water flux error       = ',werrn,
     $         ' werrn = (fluxn-delmin)/mtotn' 

          write (6,*) '----------------------------'
          write (6,903) 'arwt atm heat flux (W) = ',fhatmn,' fhatmn' 
          write (6,903) 'arwt ocn heat flux (W) = ',fhocnn,' fhocnn' 
          write (6,903) 'arwt frzl heat flux(W) = ',fhfrzn,' fhfrzn' 
          write (6,903) 'arwt tot enthalpy  (J) = ',etotn,' etotn' 
          write (6,903) 'arwt net heat      (J) = ',hnetn,' hnetn' 
          write (6,*)   '(= (fhatmn - fhocnn - fhfrzn) * dt)'
          write (6,903) 'arwt tot enth. chng(J) = ',delein,' delein' 
          write (6,904) 'arwt heat error        = ',herrn
     $         ,' herrn = (hnetn - delein) / etotn' 
          write (6,*) '----------------------------'

          call shr_sys_flush(6)

        endif
      endif

 899  format (27x,a24,2x,a24)
 900  format (a25,2x,f24.17,2x,f24.17)
 901  format (a25,2x,1pe24.17,2x,1pe24.17)
 902  format (a25,2x,1pe24.17,2x,1pe24.17)
 903  format (a25,2x,1pe24.17,2x,a10)
 904  format (a25,2x,1pe24.17,2x,a32)

      end subroutine runtime_diags

c=======================================================================

      subroutine init_mass_diags

!---!-------------------------------------------------------------------
!---! computes global combined ice and snow mass sum
!---!-------------------------------------------------------------------

      use ice_mpi_internal
      use ice_grid
      use ice_state
      use ice_mechred

      real (kind=dbl_kind) ::
     &   shmaxn, snwmxn,  shmaxs, snwmxs

      real (kind=dbl_kind) ::
     &   works(imt_local,jmt_local)

      integer (kind=int_kind) :: i,j,layer

      ! total ice volume
      call get_sum(0,tarean,one,vice,shmaxn)
      call get_sum(0,tareas,one,vice,shmaxs)
      ! total snow volume
      call get_sum(0,tarean,one,vsno,snwmxn)
      call get_sum(0,tareas,one,vsno,snwmxs)

      ! north/south ice+snow volume
      totmn = rhoi*shmaxn + rhos*snwmxn
      totms = rhoi*shmaxs + rhos*snwmxs

      ! north/south ice+snow enthalpy
      ! total ice/snow enthalpy
       do j=1,jmt_local
        do i=1,imt_local
         works(i,j) = c0
        enddo
       enddo
      do layer = 1,ntilay
       do j=1,jmt_local
        do i=1,imt_local
         works(i,j) = works(i,j) + eicen(i,j,layer) ! ice enthalpy
        enddo
       enddo
      enddo
      call get_sum(0,tarean,one,works,toten)
      call get_sum(0,tareas,one,works,totes)
      toten = toten - snwmxn * rLfs   ! plus snow enthalpy
      totes = totes - snwmxs * rLfs 

      mass2col = c0
      enth2col = c0

      end subroutine init_mass_diags

c=======================================================================

      subroutine init_diags

!---!-------------------------------------------------------------------
!---! initialize diagnostic output
!---!-------------------------------------------------------------------

      totmn  = c0      ! initial water budget term
      totms  = c0      ! initial water budget term

      if (print_points) call intpnt

      end subroutine init_diags

c=======================================================================

      subroutine intpnt

!---!-------------------------------------------------------------------
!---! initialize printout of points to fortran units
!---!-------------------------------------------------------------------

      use ice_grid

      real (kind=dbl_kind) ::
     &     minbth,     ! sum of distances in lat and lon
     $     latdis,     ! latitude distance
     $     londis,     ! longitude distance
     $     totdis      ! total distance
      integer (kind=int_kind) ::
     $             n,  ! index for search
     $             i,  ! longitude index
     $             j   ! latitude  index

      character (char_len) :: label(npnt)
      logical (kind=log_kind) :: prnt     ! for prints
      data prnt / .true. /

      ! initialize labels
      if (my_task.eq.master_task) 
     &   write(6,*) ' initialize diagnostic prints '

      label(1)(1:40)  = 'SHEBA Camp (approximately)              '
      label(2)(1:40)  = 'Near-North Pole pack ice                '
      label(3)(1:40)  = 'Bering Strait                           '
      label(4)(1:40)  = 'Southern Hudson Bay                     '
      label(5)(1:40)  = 'Icelandic Sea                           '
      label(6)(1:40)  = 'Central North Atlantic                  '
      label(7)(1:40)  = 'Antarctic Circle south of New Zealand   '
      label(8)(1:40)  = 'Weddell Sea near Antarctic Circle       '
      label(9)(1:40)  = 'Ross Sea                                '
      label(10)(1:40) = 'East Antarctica                         '
      label(11)(1:40) = 'Southern Ocean  south of Africa         '
      label(12)(1:40) = 'Central South Atlantic                  '

      ! initialize units
      do n=1,npnt
        iunit = untout(n)
        write(iunit,10) label(n)(1:40)
   10   format('.* ',a40)
        write(iunit,20) latpnt(n),lonpnt(n)
   20   format('.* lat = ',f7.2,'  lon = ',f7.2)
        write(iunit,30) 
   30   format(
     $'  in idate     z      u      v      T       q   SW    ',
     $'LW  rain  snw  SST  Qfrzmlt'/
     $' out  idate   netsw   lwup   sens   lat   aice       hi    ',
     $' hs    Tsfc    alb1   melth   meltw'/
     $' itd idate   hin1   hin2   hin3   hin4   hin5   aice0   aicen1',
     $'    aicen2    aicen3    aicen4    aicen5'/
     $'tshsn idate  Tsfcn1   Tsfcn2   Tsfcn3   Tsfcn4   Tsfcn5   hsn1',
     $'    hsn2    hsn3    hsn4    hsn5')
      enddo

      ! find indices
      if (my_task.eq.master_task) write(6,*) ' find indices of points '

      do n=1,npnt
        write(6,50) my_task,n,latpnt(n),lonpnt(n)
 50     format('task ',i4,' looking for indices for point = ',i2,
     $         ' lat lon =',2(f7.1,2x))

      ! search to see if desire to select this point:
      minbth = 540.0_dbl_kind    !  360.0 + 180.
      iindx(n)   = 0
      jindx(n)   = 0
      do j=jlo,jhi
        do i=ilo,ihi
          latdis = abs(latpnt(n)-TLAT(i,j))
          londis = abs(lonpnt(n)-TLON(i,j))
          totdis = latdis + londis
          if( totdis .lt. minbth ) then
            minbth = totdis
            jindx(n)   = j
            iindx(n)   = i
          endif
        enddo
      enddo

      if( jindx(n) .eq. 0 .or. iindx(n) .eq. 0 ) then
        write(6,1234) latpnt(n),lonpnt(n),iindx(n),jindx(n)
 1234   format(/
     $  ' .... failure to find point in fndind...'/
     $  ' .... lat lon iindx(n) jindx(n) = ',
     $   2(f7.1,1x),1x,2(i3,1x))

      else

        write(6,2234) n,latpnt(n),lonpnt(n),TLAT(iindx(n),jindx(n)),
     $                TLON(iindx(n),jindx(n)),iindx(n),jindx(n)
 2234   format(
     $  ' .... found point in fndind...'/
     $  'point    lat     lon    TLAT    TLON iindx(n) jindx(n) '/
     $   i4,1x,2(f7.1,1x),2(f7.1,1x),2(i3,8x))
      endif

      enddo

      end subroutine intpnt

c=======================================================================

      subroutine diags_cpl_in

!---!-------------------------------------------------------------------
!---! print out grid point input from coupler
!---!-------------------------------------------------------------------

      use ice_flux
      use ice_calendar

      integer (kind=int_kind) ::
     &   i, j, n                     ! generic indices
     &,  imx, jmx                  ! for diagnostic prints
      real (kind=dbl_kind) ::  
     &   rn,snw           ! for diagnostic prints

      if (print_points) then
      do j=jlo,jhi
        do i=ilo,ihi
          do n=1,npnt
           if( j .eq. jindx(n) ) then
             if( i .eq. iindx(n) ) then
               iunit = untout(n)
               rn    = (Frain(i,j)/rhow)*mps_to_cmpyr  ! cm/yr
               snw   = (Fsnow(i,j)/rhos)*mps_to_cmpyr  ! cm/yr
               write(iunit,100) idate,zlvl(i,j),uatm(i,j),vatm(i,j),
     $                Tair(i,j)-Tffresh,Qa(i,j)*c1000,
     $                Fsw(i,j),Flw(i,j),rn,snw,sst(i,j),frzmlt(i,j)
 100           format('  in',1x,i6,1x,3(f6.1,1x),f7.2,1x,f6.1,1x,
     $                4(f5.1,1x),f7.3,1x,f9.1)
               call shr_sys_flush(iunit)
             endif 
           endif
          enddo 
        enddo
      enddo 
      endif

      end subroutine diags_cpl_in

c=======================================================================

      subroutine diags_cpl_out

!---!-------------------------------------------------------------------
!---! print out grid point info sent to coupler
!---!-------------------------------------------------------------------

      use ice_flux
      use ice_albedo
      use ice_calendar
      use ice_state

      integer (kind=int_kind) ::
     &   i, j, n                      ! generic indices
      real (kind=dbl_kind) ::  
     &   hi,hs,bbalb,frsh    ! for diagnostic prints 

      if(print_points) then

      do j=jlo,jhi
        do i=ilo,ihi
          do n=1,npnt
           if( j .eq. jindx(n) ) then
             if( i .eq. iindx(n) ) then
               iunit = untout(n)
               hi = c0
               hs = c0
               if( aice(i,j) .gt. 0. ) then
                 hi = vice(i,j)/aice(i,j)
                 hs = vsno(i,j)/aice(i,j)
               endif
               bbalb = alvdf(i,j)*(awtvdr+awtvdf) 
     &               + alidf(i,j)*(awtidr+awtidf)
               frsh  = (Fresh(i,j)/rhow)*mps_to_cmpyr/c100 ! m/yr
               write(iunit,100) idate,Fswthru(i,j),Flwout(i,j),
     $               Fsensible(i,j),Flatent(i,j),aice(i,j),hi,hs,
     $               Tsfc(i,j),bbalb,Fhnet(i,j),frsh
 100           format(' out',1x,i6,1x,4(f6.1,1x),f5.3,1x,2(f7.3,1x),
     $                f6.2,1x,f5.3,1x,f7.2,1x,f6.2)
               call shr_sys_flush(iunit)
             endif
           endif
          enddo
        enddo
      enddo

      endif

      end subroutine diags_cpl_out

c=======================================================================

      subroutine print_state(plabel,i,j)

!---!-------------------------------------------------------------------
!---! this routine is useful for debugging
!---! call to it should be inserted in the form (after thermo, for example)
!---!      do j=jlo,jhi
!---!       do i=ilo,ihi
!---!         call to_column(i,j)
!---!         plabel = 'post thermo'
!---!         if (istep1.ge.check_step.and.i.eq.ip.and.j.eq.jp
!---!     &       .and.my_task.eq.mtask) 
!---!     &   call print_state(plabel,i,j)
!---!       enddo
!---!      enddo
!---! 'use ice_diagnostics' may need to be inserted also, and
!---! 'use ice_calendar' if it is not already being used
!---!
!---! author Elizabeth C. Hunke, LANL
!---!-------------------------------------------------------------------

      use ice_model_size
      use ice_kinds_mod
      use ice_calendar
      use ice_state
      use ice_itd

      character (len=20), intent(in) :: plabel
      integer (kind=int_kind), intent(in) :: i,j
      real (kind=dbl_kind) :: edebug
      integer (kind=int_kind) :: nc, layer

      write (6,*) plabel,' istep1 = ',istep1
      write (6,*) 'my_task',my_task,' i, j=',i,j 
      do nc=1,ncat

        write (6,*) ' '
        write (6,*) 'nc=',nc
        write (6,*) 'ain', ain(nc)
        write (6,*) 'vin',vin(nc)
        if (ain(nc).gt.puny) write (6,*) 'hin',vin(nc)/ain(nc)
        write (6,*) 'vsn',vsn(nc)
        write (6,*) 'Tsfn',Tsfn(nc)
        write (6,*) ' '

      enddo

      edebug = c0
      do nc = 1,ncat
        do layer = 1,nilay(nc)
          write (6,*) 'ein, cat ',nc,' layer ',layer,ein(layer,nc)
          edebug = edebug+ein(layer,nc)
        enddo
      enddo
      write (6,*) 'eice(i,j)',edebug 
      write (6,*) ' '

      end subroutine print_state

c=======================================================================

      end module ice_diagnostics

c=======================================================================
