c $Id$
c=======================================================================
!---! message passing to/from the flux coupler
!---!
!---! author Elizabeth C. Hunke
!---!
!---! Copyright, 1999.  The Regents of the University of California.
!---! This software was produced under a U.S. Government contract 
!---! (W-7405-ENG-36) by Los Alamos National Laboratory, which is 
!---! operated by the University of California for the U.S. Department 
!---! of Energy.  The U.S. Government is licensed to use, reproduce, and 
!---! distribute this software.  Permission is granted to the public to 
!---! copy and use this software without charge, provided that this 
!---! Notice and any statement of authorship are reproduced on all 
!---! copies.  Neither the Government nor the University makes any 
!---! warranty, express or implied, or assumes any liability or 
!---! responsibility for the use of this software.
c=======================================================================

      module ice_coupling

      use ice_model_size

ccxcc
ccxcc #if coupled


      implicit none

      integer (kind=int_kind), parameter ::
     &   ncbuffi = 100       ! size of integer control buffer
     &,  nsnd = 19          ! number of fields sent to coupler
     &,  nrcv = 21          ! number of fields sent from coupler
ccxcc 

     &   + 2      ! increase by 2 for closing and divergence

     &,  msgtype_c2ii = 31  ! message type for cpl->ice init
     &,  msgtype_i2ci = 40  ! message type for ice->cpl init
     &,  msgtype_c2i  = 30  ! message type for cpl->ice
     &,  msgtype_i2c  = 41  ! message type for ice->cpl
     &,  nrbuff = nrcv*imt_global*jmt_global ! size of receive buffer
     &,  nsbuff = nsnd*imt_global*jmt_global ! size of send    buffer

      integer (kind=int_kind) ::
     &   cbuffi(ncbuffi)    ! control buffer from cpl
     &,  cpl_task           ! master task id for coupler
     &,  nadv_i             ! number of coupler comms per day

      real (kind=dbl_kind) ::
     &   rbuff(imt_global,jmt_global,nrcv)  ! buffer rcvd from coupler
     &,  sbuff(imt_global,jmt_global,nsnd)  ! buffer sent to   coupler

      ! buffer location indexes
      integer (kind=int_kind), parameter ::
        !  ocean states
     &    kbufr_sst     =  1     ! sea surface temp         (K)
     &,   kbufr_sss     =  2     ! sea surface salinity     (o/oo)
     &,   kbufr_uocn    =  3     ! ocean current            (m/s)
     &,   kbufr_vocn    =  4     ! 
        !  atm states
     &,   kbufr_zlvl    =  5     ! atm level height         (m)
     &,   kbufr_uatm    =  6     ! wind                     (m/s)
     &,   kbufr_vatm    =  7     ! 
     &,   kbufr_potT    =  8     ! atm potential T          (K)
     &,   kbufr_Tair    =  9     ! atm temperature          (K) 
     &,   kbufr_Qa      = 10     ! atm specific humidity    (kg/kg)
     &,   kbufr_rhoa    = 11     ! atm air density          (kg/m^3)
        !  more ocean states
     &,   kbufr_tiltx   = 12     ! sea surface slope  
     &,   kbufr_tilty   = 13     !
        !  ocean -> ice flux
     &,   kbufr_fw      = 14     ! ptntl to form/melt ice   (W/m^2)
        !  atm -> ice fluxes
     &,   kbufr_swdidr  = 15     !          near IR,   drct
     &,   kbufr_swdvdr  = 16     ! sw down, vsbl,      drct (W/m^2)
     &,   kbufr_swdidf  = 17     !          near IR,   dffs
     &,   kbufr_swdvdf  = 18     !          vsbl,      dffs
     &,   kbufr_flw     = 19     ! longwave down            (W/m^2)
     &,   kbufr_rain    = 20     ! precip, rain             (kg/m^2 s)
     &,   kbufr_snow    = 21     ! precip, snow             (kg/m^2 s)
ccxcc

        !  external closing and divu
     &,   kbufr_clsg    = 22     ! closing rate (1/s)
     &,   kbufr_divu    = 23     ! divu (1/s)

        !  ice states
     &,   kbufs_ifrc    =  1     ! ice fraction
     &,   kbufs_tsfc    =  2     ! surface temperature      (K)
     &,   kbufs_alb1    =  3     ! albedo, visible, direct
     &,   kbufs_alb2    =  4     !         near IR, direct
     &,   kbufs_alb3    =  5     !         visible, diffuse
     &,   kbufs_alb4    =  6     !         near IR, diffuse
        !  ice -> atm fluxes
     &,   kbufs_tauxa   =  7     ! wind stress              (N/m^2)
     &,   kbufs_tauya   =  8
     &,   kbufs_lat     =  9     ! latent heat flux         (W/m^2)
     &,   kbufs_sens    = 10     ! sensible heat flux       (W/m^2)
     &,   kbufs_lwup    = 11     ! outgoing longwave radiation (W/m^2)
     &,   kbufs_evap    = 12     ! evaporated water         (kg/m^2 s)
        !  2m atm reference temperature
     &,   kbufs_Tref    = 13     ! (K)
        !  ice -> ocean fluxes
     &,   kbufs_netsw   = 14     ! penetrating shortwave -> ocean (W/m^2)
     &,   kbufs_melth   = 15     ! net ocean heat used      (W/m^2)
     &,   kbufs_meltw   = 16     ! water flux -> ocean      (kg/m^2 s)
     &,   kbufs_salt    = 17     ! salt flux -> ocean       (kg/m^2 s)
     &,   kbufs_tauxo   = 18     ! ice/ocean stress         (N/m^2)
     &,   kbufs_tauyo   = 19

c=======================================================================

      contains

c=======================================================================

      subroutine init_cpl
ccxcc

      end subroutine init_cpl

c=======================================================================

      subroutine from_coupler

!---!-------------------------------------------------------------------
!---! flux coupler -> ice data
!---!-------------------------------------------------------------------

      use ice_domain
      use ice_constants
      use ice_flux
      use ice_timers
      use ice_mpi_internal
      use ice_diagnostics
      use ice_grid
      use ice_calendar
      use ice_state
      use ice_mechred   ! ccxcc 
      use ice_ocean     ! ccxcc 

      integer (kind=int_kind) ::
     &   i, j                      ! generic indices
ccxcc


      real (kind=dbl_kind) ::  
     &   workx, worky

         call ice_timer_start(8)  ! time spent coupling

ccxcc


      !-----------------------------------------------------------------
      ! broadcast write_restart flag
      !-----------------------------------------------------------------
      if (cbuffi(22) == 1) then
        if (my_task == master_task) 
     &     write (*,*) '(from_coupler) received write restart signal'
        write_restart = 1
      endif
      call ice_bcast_iscalar(write_restart)

      !-----------------------------------------------------------------
      ! broadcast stop_now flag
      !-----------------------------------------------------------------
      if (cbuffi(3) == 1) then
        if (my_task == master_task) 
     &     write (*,*) '(from_coupler) received terminate signal'
        stop_now = 1
      endif
      call ice_bcast_iscalar(stop_now)

      !-----------------------------------------------------------------
      ! distribute data to subdomains
      !-----------------------------------------------------------------

        !  atm states                                     ! arrival units
      call global_scatter(rbuff(1,1,kbufr_zlvl)  ,zlvl )  ! m
      call global_scatter(rbuff(1,1,kbufr_uatm)  ,uatm )  ! m/s
      call global_scatter(rbuff(1,1,kbufr_vatm)  ,vatm )  ! m/s
      call global_scatter(rbuff(1,1,kbufr_potT)  ,potT )  ! K
      call global_scatter(rbuff(1,1,kbufr_Tair)  ,Tair )  ! K
      call global_scatter(rbuff(1,1,kbufr_Qa)    ,Qa   )  ! kg/kg
      call global_scatter(rbuff(1,1,kbufr_rhoa)  ,rhoa )  ! kg/m^3
        !  atm -> ice fluxes
      call global_scatter(rbuff(1,1,kbufr_swdvdr),swvdr)  ! W/m^2
      call global_scatter(rbuff(1,1,kbufr_swdvdf),swvdf)  ! W/m^2
      call global_scatter(rbuff(1,1,kbufr_swdidr),swidr)  ! W/m^2
      call global_scatter(rbuff(1,1,kbufr_swdidf),swidf)  ! W/m^2
      call global_scatter(rbuff(1,1,kbufr_flw)   ,Flw  )  ! W/m^2
      call global_scatter(rbuff(1,1,kbufr_rain)  ,Frain)  ! kg/m^2 s
      call global_scatter(rbuff(1,1,kbufr_snow)  ,Fsnow)  ! kg/m^2 s liquid
        !  ocean states
ccxcc

      call global_scatter(rbuff(1,1,kbufr_sss)   ,sss  )  ! o/oo
      call global_scatter(rbuff(1,1,kbufr_uocn)  ,uocn)   ! m/s
      call global_scatter(rbuff(1,1,kbufr_vocn)  ,vocn)   ! m/s
      call global_scatter(rbuff(1,1,kbufr_tiltx) ,strtltx)! m/m
      call global_scatter(rbuff(1,1,kbufr_tilty) ,strtlty)! m/m
ccxcc

        !  deep ocean heat flux
      call global_scatter(rbuff(1,1,kbufr_fw)    ,qdeep) ! W/m^2
        !  closing and divu
      call global_scatter(rbuff(1,1,kbufr_clsg)  ,closing)  ! 1/s
      call global_scatter(rbuff(1,1,kbufr_divu)  ,divu   )  ! 1/s


      !-----------------------------------------------------------------
      ! rotate zonal/meridional vectors to local coordinates
      ! compute data derived quantities
      !-----------------------------------------------------------------

      ! interpolate ocean dynamics variables from T-cell centers to 
      ! U-cell centers    ! NOTE ANGLE is on the U grid !
      call t2ugrid(uocn)
      call t2ugrid(vocn)
      call t2ugrid(strtltx)
      call t2ugrid(strtlty)

c$OMP PARALLEL DO PRIVATE(i,j,workx,worky)
      do j=jlo,jhi
       do i=ilo,ihi
        ! ocean
        workx      = uocn  (i,j)            ! currents, m/s 
        worky      = vocn  (i,j)
        uocn(i,j) = workx*cos(ANGLE(i,j))   ! convert to POP grid 
     1             + worky*sin(ANGLE(i,j))
        vocn(i,j) = worky*cos(ANGLE(i,j))
     1             - workx*sin(ANGLE(i,j))

        workx      = strtltx  (i,j)            ! sea sfc tilt, m/m
        worky      = strtlty  (i,j)      
        strtltx(i,j) = workx*cos(ANGLE(i,j))   ! convert to POP grid 
     1             + worky*sin(ANGLE(i,j))
        strtlty(i,j) = worky*cos(ANGLE(i,j))
     1             - workx*sin(ANGLE(i,j))

ccxcc

c        Tf    (i,j) = -1.8_dbl_kind       ! hardwired for NCOM
c        Tf    (i,j) = -depressT*sss(i,j)      ! freezing temperature (C)
        Tf    (i,j) = -depressT*max(sss(i,j),ice_ref_salinity) ! freezing T (C)
        enddo
      enddo

      ! atmo variables are needed in T cell centers in subroutine stability,
      ! and are interpolated to the U grid later as necessary
c$OMP PARALLEL DO PRIVATE(i,j,workx,worky)
      do j=jlo,jhi
       do i=ilo,ihi
        ! atmosphere
        workx      = uatm(i,j)               ! wind velocity, m/s
        worky      = vatm(i,j) 
        uatm (i,j) = workx*cos(ANGLET(i,j))   ! convert to POP grid
     1             + worky*sin(ANGLET(i,j))   ! note uatm, vatm, wind
        vatm (i,j) = worky*cos(ANGLET(i,j))   !  are on the T-grid here
     1             - workx*sin(ANGLET(i,j))

        wind (i,j) = sqrt(uatm(i,j)**2 + vatm(i,j)**2) ! wind speed, m/s

        Fsw  (i,j) = swvdr(i,j) + swvdf(i,j)
     1             + swidr(i,j) + swidf(i,j)
       enddo
      enddo

      call diags_cpl_in

      time_forc=time

      call ice_timer_stop(8)   ! time spent coupling

      end subroutine from_coupler

c=======================================================================

      subroutine to_coupler

!---!-------------------------------------------------------------------
!---! ice -> flux coupler data
!---!-------------------------------------------------------------------

      use ice_model_size
      use ice_domain
      use ice_constants
      use ice_flux
      use ice_timers
      use ice_mpi_internal
      use ice_albedo
      use ice_diagnostics
      use ice_grid
      use ice_calendar
      use ice_state

      integer (kind=int_kind) ::
     &   i, j                      ! generic indices

      real (kind=dbl_kind) ::  
     &   workx, worky              ! tmps for converting grid
     &,  Tsrf (ilo:ihi,jlo:jhi)       ! surface temperature
     &,  tauxa(ilo:ihi,jlo:jhi)       ! atmo/ice stress
     &,  tauya(ilo:ihi,jlo:jhi)               
     &,  tauxo(ilo:ihi,jlo:jhi)       ! ice/ocean stress
     &,  tauyo(ilo:ihi,jlo:jhi)               
     &,  ailohi (ilo:ihi,jlo:jhi)               

         call ice_timer_start(8)  ! time spent coupling

c$OMP PARALLEL DO PRIVATE(i,j,workx,worky)
      do j=jlo,jhi
       do i=ilo,ihi
        ! surface temperature
        Tsrf(i,j)  = Tffresh + Tsfc(i,j)                      !K

        ! wind stress  (on POP T-grid:  convert to lat-lon)
        workx = strairxT(i,j)                               ! N/m^2
        worky = strairyT(i,j)                               ! N/m^2
        tauxa(i,j) = workx*cos(ANGLET(i,j)) - worky*sin(ANGLET(i,j))
        tauya(i,j) = worky*cos(ANGLET(i,j)) + workx*sin(ANGLET(i,j))
        ! ice/ocean stress (on POP T-grid:  convert to lat-lon)
        workx = -strocnxT(i,j)                               ! N/m^2
        worky = -strocnyT(i,j)                               ! N/m^2
        tauxo(i,j) = workx*cos(ANGLET(i,j)) - worky*sin(ANGLET(i,j))
        tauyo(i,j) = worky*cos(ANGLET(i,j)) + workx*sin(ANGLET(i,j))
       enddo
      enddo

      !-----------------------------------------------------------------
      ! gather coupling variables from subdomains
      !-----------------------------------------------------------------
                                                        ! departure units 
      do j=jlo,jhi
        do i=ilo,ihi
          ailohi(i,j) = aice(i,j)
        enddo
      enddo
      call global_gather(sbuff(1,1,kbufs_ifrc), ailohi   )  ! none
      call global_gather(sbuff(1,1,kbufs_tsfc), Tsrf     )  ! K
      call global_gather(sbuff(1,1,kbufs_alb1), alvdr    )  ! none
      call global_gather(sbuff(1,1,kbufs_alb2), alidr    )  ! none
      call global_gather(sbuff(1,1,kbufs_alb3), alvdf    )  ! none
      call global_gather(sbuff(1,1,kbufs_alb4), alidf    )  ! none
      call global_gather(sbuff(1,1,kbufs_lat ), Flatent  )  ! W/m^2
      call global_gather(sbuff(1,1,kbufs_sens), Fsensible)  ! W/m^2
      call global_gather(sbuff(1,1,kbufs_lwup), Flwout   )  ! W/m^2
      call global_gather(sbuff(1,1,kbufs_evap), evap )  ! kg/m^2 s
      call global_gather(sbuff(1,1,kbufs_tauxa),tauxa)  ! N/m^2
      call global_gather(sbuff(1,1,kbufs_tauya),tauya)  ! N/m^2
      call global_gather(sbuff(1,1,kbufs_netsw),Fswthru)  ! W/m^2
      call global_gather(sbuff(1,1,kbufs_melth),Fhnet  )  ! W/m^2
      call global_gather(sbuff(1,1,kbufs_meltw),Fresh  )  ! kg/m^2 s
      call global_gather(sbuff(1,1,kbufs_salt ),Fsalt  )  ! kg/m^2 s
      call global_gather(sbuff(1,1,kbufs_tauxo),tauxo)  ! N/m^2
      call global_gather(sbuff(1,1,kbufs_tauyo),tauyo)  ! N/m^2
      call global_gather(sbuff(1,1,kbufs_Tref), Tref )  ! K

      cbuffi( 4) = idate          ! date (coded: yyyymmdd)
      cbuffi( 5) = sec            ! elapsed seconds into date
      cbuffi( 7) = imt_local      ! full size in x-dir of ice grid
      cbuffi( 8) = jmt_local      ! full size in y-dir of ice grid
      cbuffi( 9) = nadv_i         ! ice comm pairs per day

ccxcc


      call diags_cpl_out

      call ice_timer_stop(8)   ! time spent coupling

      end subroutine to_coupler

ccxcc



c=======================================================================

      end module ice_coupling

c=======================================================================
