c $Id$
c=======================================================================
!---! ocean boundary interface (stability based flux calculations)
!---! for uncoupled runs and for coupled runs
!---!
!---! author John Weatherly, C.M. Bitz,
!---!        Elizabeth C. Hunke, Bruce P. Briegleb
c=======================================================================
#ifdef column
ccxcc The column model adds coupled variables to this module
ccxcc so they may be readily sent to the coupler, output file,
ccxcc or atmosphere model. These variables should be initialized
ccxcc at the start of each timestep with subroutine init_ocn. The 
ccxcc column model uses subroutine mixed_layer with slight 
ccxcc modifications so that it calls ocncolumn, a unique routine 
ccxcc for computing sst and frzmlt for the column model. The key 
ccxcc differences being that ocncolumn uses the variables that 
ccxcc are now defined for this module in column mode and
ccxcc it uses a zenith angle and wavelength dependent ocean albedo.
#endif
      module ice_ocean

      use ice_flux
      use ice_calendar

      implicit none

      real (kind=dbl_kind), parameter ::
     &   cpw  = 4.19e6_dbl_kind      ! Specific heat of ocean water
     &,  hmix = 20._dbl_kind     ! ocean mixed layer depth (m)
     &,  cphm = cpw*hmix

      real (kind=dbl_kind) ::
     &   sst_mixedlayer(ilo:ihi,jlo:jhi)     ! save value of sst

#ifdef column
ccxcc ocean coupling variables
      ! albedos over ocean
      real (kind=dbl_kind) ::
     &   alocvdr  (ilo:ihi,jlo:jhi)  ! visible, direct   (fraction)
     &,  alocidr  (ilo:ihi,jlo:jhi)  ! near-ir, direct   (fraction)
     &,  alocvdf  (ilo:ihi,jlo:jhi)  ! visible, diffuse  (fraction)
     &,  alocidf  (ilo:ihi,jlo:jhi)  ! near-ir, diffuse  (fraction)

c     ! variables sent to coupler for ocean in ice model
      real (kind=dbl_kind) ::
     &   fshoc    (ilo:ihi,jlo:jhi)  ! sensible         heat flux  (W/m**2)
     &,  flhoc    (ilo:ihi,jlo:jhi)  ! latent           heat flux  (W/m**2)
     &,  flwupoc  (ilo:ihi,jlo:jhi)  ! long-wave upward heat flux  (W/m**2)
     &,  trefoc   (ilo:ihi,jlo:jhi)  ! reference temperature over ocn 
     &,  strxoc   (ilo:ihi,jlo:jhi)  ! air stress over ocn x-dir
     &,  stryoc   (ilo:ihi,jlo:jhi)  ! air stress over ocn y-dir

c     ! variables received from coupler for ocean in ice model
      real (kind=dbl_kind) ::
     &   cosznth  (ilo:ihi,jlo:jhi)  ! cosine of zenith angle
     &,  qdeep    (ilo:ihi,jlo:jhi)  ! deep ocean heat flux
#endif

c=======================================================================

      contains

c=======================================================================

      subroutine mixed_layer(i,j)

      use ice_grid
      use ice_atmo

      integer (kind=int_kind), intent(in) :: i,j
      real (kind=dbl_kind) ::
     &   delt   ! potential T difference   (K)
     &,  delq   ! humidity difference      (kg/kg)
     &,  rdn_in  ! initial value for rdn in stability routine
     &,  vmag   ! surface wind magnitude   (m/s)
     &,  strx   ! dummy argument
     &,  stry   ! dummy argument
     &,  Trf   ! dummy argument
     &,  dssqdt   ! dummy argument

      real (kind=dbl_kind), parameter ::
     &   umin =  c1          ! minimum wind speed (m/s)

        if (istep.gt.1) sst(i,j) = sst_mixedlayer(i,j)

        if (tmask(i,j)) then

         vmag   = max(umin, wind(i,j))      ! CSM Ocean wind-speed formula 
         rdn_in = sqrt(0.0027_dbl_kind/vmag 
     &     + .000142_dbl_kind + .0000764_dbl_kind*vmag) ! for rdn_in

#ifndef column
         call stability( i,j,1, sst(i,j), rdn_in,
     $         strx   ,stry   ,Trf, dssqdt,  delt, delq)

         call ocnheat(i,j, delt, delq)  ! update sst, determine frzmlt
#else
ccxcc column model uses the variables defined above
ccxcc instead of dummies so they are saved
         call stability( i,j,1, sst(i,j), rdn_in,
     &         strxoc(i,j),stryoc(i,j),trefoc(i,j), dssqdt,  delt, delq)

         call ocncolumn(i,j, delt, delq)  ! uncoupled update sst, determine frzmlt
#endif

        endif

      end subroutine mixed_layer

c=======================================================================

      subroutine ocnheat(i,j, delt, delq)

!---!-------------------------------------------------------------------
!---! update sst and freezing/melting potential 
!---!-------------------------------------------------------------------

      use ice_constants
      use ice_state
      use ice_albedo

      integer (kind=int_kind) :: i,j

      real (kind=dbl_kind), intent(in) ::
     &   delt   ! potential T difference   (K)
     &,  delq   ! humidity difference      (kg/kg)

      real (kind=dbl_kind) ::
     &   TsfK   ! surface temperature in Kelvin (K)
     &,  fsh    ! sensible         heat flux  (W/m**2)
     &,  flh    ! latent           heat flux  (W/m**2)
     &,  fswabs ! srfc absrbd short-wave heat flux (W/m**2)
     &,  flwup  ! long-wave upward heat flux  (W/m**2)

      real (kind=dbl_kind) ::
     &   ft     ! fraction reduction of positive qdp
     &,  qdp    ! deep ocean heat flux

      ! specify as constant for now
      qdp   = -c10

      ! ocean surface temperature in Kelvin
      TsfK = sst(i,j)+Tffresh

      ! shortwave radiative flux
      fswabs = (c1 - albocn) * (swvdr(i,j)
     & + swvdf(i,j) + swidr(i,j) + swidf(i,j))

      ! longwave radiative flux
      flwup  = -emissivity*stefan_boltzmann * TsfK**4

      ! downward latent and sensible heat fluxes
      flh = lhcoef(i,j,1) * delq  ! using lhcoef(i,j,1), shcoef(i,j,1)
      fsh = shcoef(i,j,1) * delt  ! as temporary arrays for ocean

      ! first, compute sst change due to exchange with atm/ice above
      sst(i,j) = sst(i,j) +((fsh + flh + flwup + Flw(i,j) + fswabs)
     &            *(c1-aice(i,j)))*dt/cphm

      ! adjust qdp if cooling of mixed layer will occur when sst < sst0;
      ! Reduce qdp > 0 down to zero
      if( sst(i,j) .lt. 0. .and. qdp .gt. 0. ) then
        ft  = max((c1-sst(i,j)/Tf(i,j)),c0)
        qdp = qdp*ft
      endif

      ! computed T change due to exchange with deep layers:
      sst(i,j) = sst(i,j) - qdp*dt/cphm

      ! compute potential to freeze or melt ice
      frzmlt(i,j) = (Tf(i,j)-sst(i,j))*cphm/dt          

      ! if sst is below freezing, reset sst to Tf
      if(sst(i,j) .le. Tf(i,j)) sst(i,j) = Tf(i,j)

      ! save mixed layer sst
      sst_mixedlayer(i,j) = sst(i,j)

      end subroutine ocnheat

#ifdef column
c=======================================================================

      subroutine init_ocn

!---!-------------------------------------------------------------------
!---! initialize fields sent to coupler for ocean in column model
!---!-------------------------------------------------------------------

      use ice_constants

      integer i,j

      do j=jlo,jhi
        do i=ilo,ihi

        Fshoc(i,j)  = c0
        Flhoc(i,j)  = c0
        Flwupoc(i,j)= c0
        strocnxT(i,j) = rhow *uocn(i,j)*uocn(i,j) ! ocean stress, T grid
        strocnyT(i,j) = c0
        strxoc(i,j) = c0                          ! wind stress, T grid
        stryoc(i,j) = c0
        Trefoc(i,j) = Tair(i,j)  

        enddo
      enddo

      end subroutine init_ocn

c=======================================================================

      subroutine ocn_albedos

!---!-------------------------------------------------------------------
!---! Ice-free ocean albedos function of solar zenith angle only, and
!---! independent of spectral interval:
!---!-------------------------------------------------------------------

      use ice_constants

      integer i,j

      do j=jlo,jhi
        do i=ilo,ihi

          if (cosznth(i,j)>0.) then
            alocvdr(i,j)  = (.026/(cosznth(i,j)**1.7 + .065)) + 
     &           (.15*(cosznth(i,j) - p1)*
     &           (cosznth(i,j) - p5)*(cosznth(i,j) - c1))
            alocidr(i,j)  = alocvdr(i,j)
            alocvdf(i,j) = 0.06
            alocidf(i,j) = 0.06
          else
            alocvdr(i,j) = c0
            alocidr(i,j) = c0
            alocvdf(i,j) = c0
            alocidf(i,j) = c0
          end if
      
        enddo
      enddo

      end subroutine ocn_albedos

c=======================================================================

      subroutine ocncolumn(i,j, delt, delq ) 

!---!-------------------------------------------------------------------
!---! update sst and freezing potential frzmlt for column model
!---!-------------------------------------------------------------------

      use ice_constants
      use ice_state

      integer (kind=int_kind) :: i,j

      real (kind=dbl_kind), intent(in) ::
     &   delt   ! potential T difference   (K)
     &,  delq   ! humidity difference      (kg/kg)

      real (kind=dbl_kind) ::
     &   TsfK  ! surface temperature in Kelvin (K)
     &,  fswabs     ! srfc absrbd short-wave heat flux (W/m**2)

      ! ocean surface temperature in Kelvin
      TsfK = sst(i,j)+Tffresh

      ! shortwave radiative flux
      fswabs = (c1 - alocvdr(i,j)) * swvdr(i,j)
     &       + (c1 - alocidr(i,j)) * swvdf(i,j)
     &       + (c1 - alocvdf(i,j)) * swidr(i,j) 
     &       + (c1 - alocidf(i,j)) * swidf(i,j)

      ! longwave radiative flux
      flwupoc(i,j)   = -emissivity*stefan_boltzmann * TsfK**4

      ! downward latent and sensible heat fluxes
      flhoc(i,j) = lhcoef(i,j,1) * delq  ! using lhcoef(i,j,1), shcoef(i,j,1)
      fshoc(i,j) = shcoef(i,j,1) * delt  ! as temporary arrays for ocean

      ! ocean mixed layer temperature
      sst(i,j) = sst(i,j) +((fshoc(i,j) + flhoc(i,j) + flwupoc(i,j) 
     &           + Flw(i,j) + fswabs)*(c1-aice(i,j)) 
     &           - qdeep(i,j))*dt/cphm

      ! compute potential to freeze or melt ice
      frzmlt(i,j) = (Tf(i,j)-sst(i,j))*cphm/dt          

      ! if sst is below freezing, reset sst to Tf
      if(sst(i,j) .le. Tf(i,j)) sst(i,j) = Tf(i,j)

      ! save mixed layer sst
      sst_mixedlayer(i,j) = sst(i,j)

      end subroutine ocncolumn

#endif
c=======================================================================

      end module ice_ocean

c=======================================================================
