c $Id: ice_itd.F,v 1.2 2001/02/23 21:06:54 schramm Exp $
c=======================================================================
!---! Routines to initialize the ice thickness distribution and 
!---! utilities to redistribute ice among categories. These routines 
!---! are not specific to a particular numerical implementation.
!---!
!---! author C. M. Bitz
!---!
!---! See Bitz, C.M., and W.H. Lipscomb, 1999: 
!---! An energy-conserving thermodynamic model of sea ice,
!---! J. Geophys. Res., 104, 15,669-15,677. 
!---!     
!---! See Bitz, C.M., M.M. Holland, A.J. Weaver, M. Eby:
!---! Simulating the ice-thickness distribution in a climate model
!---! accepted to J. Geophys. Res.
!---! (prepublications copy available upon request)
!---!
!---! The author grants permission to the public to copy and use this
!---! software without charge, provided that this Notice and any statement
!---! of authorship are reproduced on all copies and any publications that
!---! result from the use of this software must (1) refer to the publications 
!---! listed above and (2) acknowledge the origin and author of the model.
!---! This software is without warranty, expressed or implied, and the
!---! author assumes no liability or responsibility for its use. 
c=======================================================================
!---! code heavily modified by Elizabeth C. Hunke, LANL
c=======================================================================

      module ice_itd

      use ice_kinds_mod
      use ice_model_size
      use ice_constants
      use ice_state

      implicit none

      integer (kind=int_kind) ::
     &   kitd    ! type of itd conversions 
                 !   0 = delta function
                 !   1 = linear remap  !! not implemented
     &,  nilay    (ncat)      ! the number of layers in each cat
     &,  layer1   (ncat)      ! position of the top layer in each cat
     &,  layern   (ncat)      ! position of the bottom layer in each cat

      real (kind=dbl_kind), parameter ::
     &   hi_min = 0.1_dbl_kind         ! minimum ice thickness allowed (m)

      real (kind=dbl_kind) ::
     &   hs_min               ! minimum snow thickness allowed (m)
     &,  hin_max  (0:ncat)    ! category limits                (m)
     &,  hin_mp   (0:ncat)    ! midpoint of category limits    (m)
     &,  ain_min  (0:ncat)    ! minimum fract. area allowed in each cat

!---!-------------------------------------------------------------------
!---! a note regarding hi_min and hin_max(0):
!---! both represent a minimum ice thickness.  hin_max(0) is
!---! intended to be used for particular numerical implementations
!---! of category conversions in the ice thickness distribution.
!---! hi_min is a more general purpose parameter, but is specifically 
!---! for maintaining stability in the thermodynamics.  Currently,
!---! hi_min = 0.1 m
!---! hin_max(0) = 0.1 m for the delta function itd
!---! hin_max(0) = 0.0 m for linear remapping
!---!
!---! similarly, there are two values of minimum snow thickness
!---! (the other is defined in ice_vthermo.H since it is used only
!---! for thermo.) 
!---!
!---! Also note that the upper limit on the thickest category
!---! is only used for the linear remapping scheme
!---! and it is not a true upper limit on the thickness
!---!-------------------------------------------------------------------

c=======================================================================

      contains

c=======================================================================

      subroutine init_itd

!---!-------------------------------------------------------------------
!---! Initialize area fraction and thickness boundaries for the itd model
!---!
!---! authors William H. Lipscomb
!---!         Elizabeth C. Hunke and C. M. Bitz
!---!-------------------------------------------------------------------

      integer (kind=int_kind) :: nc
      real (kind=dbl_kind):: cc1, cc2, cc3, x1

      if ((ncat.eq.1) .and. (kitd.eq.1)) then
        write (6,*) 'Remapping the ITD is not allowed for ncat=1.'
        write (6,*) 'Use the delta function ITD option instead.'
        stop
      endif

      ! minimum allowable fract
      ! test carefully when changing these values
      ! round off error can cause serious problems if
      ! ain_min is too small but it should be as small
      ! as possible for accuracy
      ain_min(0)=1.0e-6_dbl_kind
      do nc=1,ncat
        ain_min(nc)=5.0e-6_dbl_kind
      enddo

      ! minimum allowable snow thickness
c      hs_min = 0.00001_dbl_kind
      hs_min = puny

      if (kitd.eq.1) then
        ! linear remapping itd category limits
        cc1 = c3/real(ncat)
        cc2 = c15*cc1      
        cc3 = c3

        hin_max(0) = c0  ! minimum ice thickness, m
      else
        ! delta function itd category limits
        cc1 = max(1.1_dbl_kind/ncat,c1*hi_min)
        cc2 = c25*cc1
        cc3 = 2.25_dbl_kind

        ! hin_max(0) should not be zero 
        ! use some caution in making it less than 0.10
        hin_max(0) = hi_min  ! minimum ice thickness, m
      endif

      do nc = 1, ncat
       x1 = real(nc-1) / real(ncat)
       hin_max(nc) = hin_max(nc-1) + cc1 + cc2*(c1 + tanh(cc3*(x1-c1)))
      enddo

      do nc = 1, ncat
        hin_mp(nc) = p5*( hin_max(nc-1) + hin_max(nc) ) 
      enddo

      if (my_task.eq.master_task) then
      write (6,*) ''
      write (6,*) 'hin_max(nc-1) < Cat nc < hin_max(nc)'
      do nc = 1, ncat
         write (6,*) hin_max(nc-1),' < Cat ', nc, ' < ', hin_max(nc)
      enddo
      endif

      end subroutine init_itd

c=======================================================================

      subroutine aggregate

!---!-------------------------------------------------------------------
!---! aggregate ice state over the grid
!---!
!---! author C. M. Bitz
!---!-------------------------------------------------------------------

      use ice_domain
      use ice_flux
      use ice_grid

      integer (kind=int_kind) :: i, j, nc

      do j=jlo,jhi
        do i=ilo,ihi
          aice0(i,j) = c1
          aice(i,j) = c0
          vice(i,j) = c0
          vsno(i,j) = c0
          Tsfc(i,j) = Tf(i,j)
          if (tmask(i,j)) then
            Tsfc(i,j) = c0
            do nc = 1,ncat
               aice(i,j) = aice(i,j) + aicen(i,j,nc)
               vice(i,j) = vice(i,j) + vicen(i,j,nc)
               vsno(i,j) = vsno(i,j) + vsnon(i,j,nc)
               Tsfc(i,j) = Tsfc(i,j) + Tsfcn(i,j,nc)*aicen(i,j,nc)
            enddo
            if (aice(i,j).gt.0.)  then
              aice0(i,j)=c1-aice(i,j)
              Tsfc(i,j) = Tsfc(i,j)/aice(i,j)
            else
              Tsfc(i,j) = Tf(i,j)
            endif
          endif ! tmask
        enddo   ! end of loop over i,j
      enddo

      call bound(aice0)
      call bound(aice)
      call bound(vice)
      call bound(vsno)
      call bound(Tsfc)

      end subroutine aggregate

c=======================================================================

      subroutine aggregate_pt(i,j)

!---!-------------------------------------------------------------------
!---! aggregate ice thickness distribution state 
!---!
!---! author C. M. Bitz
!---!-------------------------------------------------------------------

      use ice_flux

      integer (kind=int_kind), intent(in) :: i, j
      integer (kind=int_kind) :: nc

            aice(i,j) = c0
            vice(i,j) = c0
            vsno(i,j) = c0
            Tsfc(i,j) = c0
            do nc = 1,ncat
               aice(i,j) = aice(i,j) + ain(nc)
               vice(i,j) = vice(i,j) + vin(nc)
               vsno(i,j) = vsno(i,j) + vsn(nc)
               Tsfc(i,j) = Tsfc(i,j) + Tsfn(nc)*ain(nc)
            enddo
            if (aice(i,j).gt.0.)  then
              aice0(i,j)= c1-aice(i,j)
              Tsfc(i,j) = Tsfc(i,j)/aice(i,j)
            else
              Tsfc(i,j) = Tf(i,j)
            endif

      end subroutine aggregate_pt

c=======================================================================

      subroutine to_column(i,j)

!---!-------------------------------------------------------------------
!---! define column state variables for given point (i,j)
!---!
!---! authors Elizabeth C. Hunke, LANL
!---!        and C. M. Bitz
!---!-------------------------------------------------------------------

      integer (kind=int_kind), intent(in) :: i,j
      integer (kind=int_kind) :: nc, layer

        ai0 = aice0(i,j)
        do nc = 1,ncat
          ain(nc)  = aicen(i,j,nc)
          vin(nc)  = vicen(i,j,nc)
          vsn(nc)  = vsnon(i,j,nc)
          Tsfn(nc) = Tsfcn(i,j,nc)
          do layer = 1,nilay(nc)
            ein(layer,nc) = eicen(i,j,layer1(nc)+layer-1)
          enddo
        enddo

      end subroutine to_column

c=======================================================================

      subroutine from_column(i,j)

!---!-------------------------------------------------------------------
!---! reload state variables for given point (i,j) from column variables
!---!
!---! authors Elizabeth C. Hunke, LANL
!---!        and C. M. Bitz
!---!-------------------------------------------------------------------

      integer (kind=int_kind), intent(in) :: i,j
      integer (kind=int_kind) :: nc, layer

        aice0(i,j) = ai0
        do nc = 1,ncat
          aicen(i,j,nc) = ain(nc)  
          vicen(i,j,nc) = vin(nc)  
          vsnon(i,j,nc) = vsn(nc)  
          Tsfcn(i,j,nc) = Tsfn(nc) 
          do layer = 1,nilay(nc)
             eicen(i,j,layer1(nc)+layer-1) = ein(layer,nc)
          enddo
        enddo

      end subroutine from_column

c=======================================================================

      subroutine check_state(i,j)

!---!-------------------------------------------------------------------
!---!  Insist that certain fields are monotone.
!---!  Should not be necessary if all is well, 
!---!  but best to keep going. Model will not conserve
!---!  energy and water if fields are zeroed here.
!---!
!---! author C. M. Bitz
!---!-------------------------------------------------------------------

      use ice_flux

      integer (kind=int_kind) :: i,j,layer,nc
      logical (kind=log_kind) :: zerout

      do nc=1,ncat

         zerout=.false.
         if ( (aicen(i,j,nc) .lt. 0.0).or.(vicen(i,j,nc) .lt. 0.0) )
     $        zerout = .true.
         do layer=layer1(nc),layern(nc)
            if (eicen(i,j,layer) .gt. 0.0) zerout=.true.
         enddo

         if (zerout) then

           do layer = layer1(nc),layern(nc)
             eicen(i,j,layer) = c0
           enddo

           aice0(i,j) = aice0(i,j)+aicen(i,j,nc) 
           aicen(i,j,nc) = c0
           vicen(i,j,nc) = c0
           vsnon(i,j,nc) = c0
           Tsfcn(i,j,nc) = Tf(i,j)

         elseif (vsnon(i,j,nc) .lt. 0. ) then

           vsnon(i,j,nc) = c0

         endif

      enddo ! categories

      end subroutine check_state

c=======================================================================

      subroutine distr_check

!---!-------------------------------------------------------------------
!---! Force ice thickness distribution to maintain two rules
!---! (1) the fractional area cannot be less than some limiting value
!---! (2) each categories thickness lies withing the 
!---!     max and min thickness range for that category 
!---! 
!---! author C. M. Bitz
!---!-------------------------------------------------------------------

      use ice_flux
      use ice_grid

      real (kind=dbl_kind) :: 
     &   dvicen(1), dvsnon(1)      ! total nuked ice/snow volumes north 
     &,  dvices(1), dvsnos(1)      ! total nuked ice/snow volumes south

      integer (kind=int_kind) :: i, j, nc, nr, layer
      real (kind=dbl_kind) :: 
     &   hin    (ncat)       ! ice thickness for each cat  (m)
     &,  hsn    (ncat)       ! snow thickness for each cat (m)

      logical vnorm            ! if true, renormalize ice and snow to 
      data vnorm  / .true. /   ! compensate for nuking ice/snow volumes 

      logical testitd
      data testitd  / .true. / ! test the itd
      logical gerror 
      integer (kind=int_kind) :: noffend

      dvicen(1) = c0 
      dvsnon(1) = c0 
      dvices(1) = c0 
      dvsnos(1) = c0 

      do j=jlo,jhi
        do i=ilo,ihi
          if (tmask(i,j)) then

            call to_column(i,j)

      if (ncat.gt.1) then
      !-----------------------------------------------------------------
      ! get rid of small positive area by moving to other cat (if possible)
      ! by first moving small bits of ice from thin to thicker cats
      ! and then by starting from the thickest and working down
      ! hsn and hin are dummies here
      !-----------------------------------------------------------------

            do nc = 1,ncat-1
              if ( (ain(nc) .le. ain_min(nc)) .and. 
     $             (ain(nc) .gt. 0.0       ) ) then

             call moveup( Tf(i,j),ain(nc)   ,vin(nc)   ,
     $                  vsn(nc)   ,ein(:,nc)   ,Tsfn(nc) ,
     $                  hin(nc)       ,hsn(nc)       ,hin_max(nc-1)   ,
     $               nilay(nc)     ,ain(nc+1) ,vin(nc+1) ,
     $                vsn(nc+1) ,ein(:,nc+1) ,Tsfn(nc+1),
     $                  hin(nc+1)     ,hsn(nc+1)     ,hin_max(nc)     ,
     $                  nilay(nc+1)   )
                
              endif
            enddo
      
            do nc = ncat,2,-1
              if (  (ain(nc) .le. ain_min(nc) ) .and. 
     $              (ain(nc) .gt. 0.0        )  ) then

              call movedn( Tf(i,j),ain(nc-1) ,vin(nc-1) ,
     $                vsn(nc-1) ,ein(:,nc-1) ,Tsfn(nc-1),
     $                  hin(nc-1)     ,hsn(nc-1)     ,hin_max(nc-2)   ,
     $               nilay(nc-1)   ,ain(nc)   ,vin(nc)   ,  
     $                 vsn(nc)   ,ein(:,nc)   ,Tsfn(nc) ,
     $                  hin(nc)       ,hsn(nc)       ,hin_max(nc-1)   ,
     $                  nilay(nc)     )

              endif
            enddo

      endif ! ncat > 1

      !-----------------------------------------------------------------
      !     Put tiny amounts of open water area out of misery by 
      !     adding to the thinnest ice provided it has ice already.
      !     If the open water fraction is negative (prob. never happens)
      !     then reduce the frac of the thinnest ice first. 
      !-----------------------------------------------------------------

            ai0 = c1
            do nc = 1,ncat
              ai0 = ai0-max(ain(nc),c0)
            enddo
            if (ai0.lt.ain_min(0)) then
              if (ai0.ge.0.) then 
                do nc = 1,ncat
                  if (ain(nc).gt.0.) then
                    ain(nc) = ain(nc)+ai0
                    ai0 = c0
                  endif
                enddo
              else                ! open water fract is negative
                nr = 1            ! assume can decrease frac of cat 1
                do nc = 2,ncat
                  ! make sure the frac that will decrease remains > min
                  ! if not, assume next cat will decrease 
                  if ((ain(nr)+ai0).lt.ain_min(nr)) nr = nc
                enddo
                ain(nr) = ain(nr)+ai0
                ai0 = c0
              endif
            endif

      !-----------------------------------------------------------------
      ! remove negative or small ice area/volumes
      ! the water balance may be maintained by renormalizing the 
      ! remaining ice/snow below in which case there is some slop in the 
      ! energy budget
      !-----------------------------------------------------------------

            do nc = 1,ncat
              if ( (ain(nc) .lt. ain_min(nc) ) .or. 
     $             (vin(nc) .lt. puny  ) ) then

                do layer = 1,nilay(nc)
                   ein(layer,nc) = c0
                enddo

            ! vin is the volume of ice that is removed from ice model
            ! for simplicity think of it as small but positive
            ! thankfully this works if vicen is negative too, which is 
            ! possible for certain advection schemes

                dvicen(1) = dvicen(1) + vin(nc) * tarean(i,j)
                dvsnon(1) = dvsnon(1) + vsn(nc) * tarean(i,j)
                dvices(1) = dvices(1) + vin(nc) * tareas(i,j)
                dvsnos(1) = dvsnos(1) + vsn(nc) * tareas(i,j)
                ain(nc) = c0
                vin(nc) = c0
                vsn(nc) = c0
                Tsfn(nc) = Tf(i,j)

              elseif (vsn(nc) .lt. hs_min*ain(nc)) then
                dvsnon(1) = dvsnon(1) + vsn(nc) * tarean(i,j)
                dvsnos(1) = dvsnos(1) + vsn(nc) * tareas(i,j)
                vsn(nc) = c0
              endif
            enddo

            call rebin_ice(i,j)
            call from_column(i,j)

          endif ! tmask

        enddo  
      enddo

      if (vnorm) then 
        call normalize_state(dvicen,dvices,dvsnon,dvsnos)

        do j=jlo,jhi
        do i=ilo,ihi
           call to_column(i,j) 
           call rebin_ice(i,j)

           if (testitd) then
           gerror=.false.
           noffend=0
           ai0 = c1
           do  nc=1,ncat
              ai0 = ai0-ain(nc)
              if (ain(nc) .gt. puny  ) then
                if (ain(nc) .le. ain_min(nc) ) gerror=.true.
                if (ain(nc) .le. ain_min(nc) ) noffend=nc
                hin(nc)=vin(nc)/ain(nc)
                if (hin(nc).gt.hin_max(nc)+puny.and.(nc.ne.ncat)) then
                  gerror=.true.
                  noffend=nc
                endif
                if (hin(nc).lt.hin_max(nc-1)-puny) then
                  gerror=.true.
                  noffend=nc
                endif
              endif
           enddo
           if ((ai0.lt.ain_min(0)).and.(ai0.gt.puny))  gerror=.true.
           if (gerror) then
             if (noffend.eq.0) then
               write(*,*) 'g(h) ERROR',0, ai0, 0.
             else
               write(*,*) 'g(h) ERROR',noffend,ain(noffend),hin(noffend)
             endif
           endif
           endif

          call from_column(i,j) 
        enddo
        enddo
      endif

      end subroutine distr_check

c=======================================================================

      subroutine normalize_state(dvicen,dvices,dvsnon,dvsnos)

!---!-------------------------------------------------------------------
!---! normalize the state variables for the ice thickness distribution
!---! to conserve volume after removing small areas
!---!
!---! author C. M. Bitz
!---!-------------------------------------------------------------------

      use ice_model_size
      use ice_mpi_internal
      use ice_grid
      use ice_calendar

      real (kind=dbl_kind), intent(in) :: 
     &   dvicen(1), dvsnon(1)      ! total nuked ice/snow volumes north 
     &,  dvices(1), dvsnos(1)      ! total nuked ice/snow volumes south

      integer (kind=int_kind) :: i, j, nc, layer

      real (kind=dbl_kind) :: 
     &   dviceng, dvsnong      ! total nuked ice/snow volumes north
     &,  dvicesg, dvsnosg      ! total nuked ice/snow volumes south
     &,  vitotn,  vstotn       ! total ice/snow volumes north 
     &,  vitots,  vstots       ! total ice/snow volumes south
     &,  Ricen,  Rices       ! normalization fact for nuked ice volumes
     &,  Rsnon,  Rsnos       ! normalization fact for nuked sno volumes
     &,  work1(ilo:ihi,jlo:jhi),work2(ilo:ihi,jlo:jhi)

      !-----------------------------------------------------------------
      ! aggregate volumes
      !-----------------------------------------------------------------

c$OMP PARALLEL DO PRIVATE(i,j,nc)
      do j=jlo,jhi
       do i=ilo,ihi
         if (tmask(i,j)) then
          vice(i,j) = c0
          vsno(i,j) = c0
          do nc = 1,ncat
            vice(i,j) = vice(i,j) + vicen(i,j,nc)
            vsno(i,j) = vsno(i,j) + vsnon(i,j,nc)
          enddo
         endif
       enddo
      enddo

      !-----------------------------------------------------------------
      ! total ice volume
      !-----------------------------------------------------------------

      call get_sum(0,tarean,one,vice,vitotn)
      call get_sum(0,tareas,one,vice,vitots)

      !-----------------------------------------------------------------
      ! total snow volume
      !-----------------------------------------------------------------

      call get_sum(0,tarean,one,vsno,vstotn)
      call get_sum(0,tareas,one,vsno,vstots)

      !-----------------------------------------------------------------
      ! Renormalize the ice and snow volumes and enthalpy 
      ! to account for nuked volumes
      !-----------------------------------------------------------------

      Ricen = c1
      Rices = c1
      Rsnon = c1
      Rsnos = c1

      dviceng = ice_global_real_sum(1,dvicen)
      dvicesg = ice_global_real_sum(1,dvices)
      dvsnong = ice_global_real_sum(1,dvsnon)
      dvsnosg = ice_global_real_sum(1,dvsnos)

      if (dviceng+dvicesg+dvsnong+dvsnosg.gt.puny) then
      if (vitotn.gt.0.) Ricen = Ricen + dviceng/vitotn
      if (vitots.gt.0.) Rices = Rices + dvicesg/vitots
      if (vstotn.gt.0.) Rsnon = Rsnon + dvsnong/vstotn
      if (vstots.gt.0.) Rsnos = Rsnos + dvsnosg/vstots

       do j=jlo,jhi
        do i=ilo,ihi
         work1(i,j) = Ricen*mask_n(i,j) + Rices*mask_s(i,j)
         work2(i,j) = Rsnon*mask_n(i,j) + Rsnos*mask_s(i,j)
         do nc = 1,ncat
          vicen(i,j,nc) = vicen(i,j,nc) * work1(i,j)
          vsnon(i,j,nc) = vsnon(i,j,nc) * work2(i,j)
         enddo
         do layer = 1,ntilay
          eicen(i,j,layer) = eicen(i,j,layer) * work1(i,j)
         enddo
        enddo
       enddo
      endif

      if (mod(istep1,diagfreq).eq.0.and.my_task.eq.master_task) 
     &  write(6,4000) Ricen, Rices, Rsnon, Rsnos
 4000 format('Rnormv ',4((1pe20.13,1x)))

      end subroutine normalize_state

c=======================================================================

      subroutine rebin_ice(i,j)

!---!-------------------------------------------------------------------
!---! Rebins thicknesses into defined categories
!---!
!---! author C. M. Bitz
!---!-------------------------------------------------------------------

      use ice_flux

      integer (kind=int_kind), intent(in) ::
     &   i,j

      integer (kind=int_kind) ::
     &   nc
      real (kind=dbl_kind) ::
     &   hin(ncat)          ! ice thickness for each cat        (m)
     &,  hsn(ncat)          ! ice thickness for each cat        (m)

      if (ncat.gt.1) then

      !-----------------------------------------------------------------
      ! Update thicknesses
      !-----------------------------------------------------------------
      do nc = 1, ncat
          hin(nc) = c0
          hsn(nc) = c0
          if (ain(nc).gt.eps11) then
            hin(nc) = vin(nc)/ain(nc)
            hsn(nc) = vsn(nc)/ain(nc)
          endif
      enddo
      ! make sure thickness of cat 1 is at least hin_max(0)
      if ((hin(1).lt.hin_max(0)).and.(hin_max(0).gt.c0)) then
        ain(1) = vin(1)/hin_max(0)
        hin(1) = hin_max(0)
      endif

      !-----------------------------------------------------------------
      ! see if cat 1 has outgrown its thickness limit
      !     if so, add it to cat 2, and so forth
      !-----------------------------------------------------------------
          do nc = 1,(ncat-1)      
           if (hin(nc).ge.hin_max(nc)) then
            call moveup( Tf(i,j)
     $             , ain(nc),  vin(nc),   vsn(nc)
     $                 , ein(:,nc),  Tsfn(nc)
     &          , hin(nc) , hsn(nc)
     $                 ,       hin_max(nc-1),   nilay(nc)
     $             , ain(nc+1),vin(nc+1), vsn(nc+1)
     $                 , ein(:,nc+1),Tsfn(nc+1)
     &          , hin(nc+1) , hsn(nc+1)
     $                 ,     hin_max(nc),     nilay(nc+1) )
           endif
          enddo
          
      !-----------------------------------------------------------------
      ! see if the thickest cat has become too thin (and so forth)
      !-----------------------------------------------------------------
          do nc = ncat,2,-1
           if (hin(nc).lt.hin_max(nc-1)) then
            call movedn( Tf(i,j)
     $             , ain(nc-1), vin(nc-1), vsn(nc-1)
     $                 , ein(:,nc-1), Tsfn(nc-1)
     &        , hin(nc-1), hsn(nc-1)
     $                 ,      hin_max(nc-2),   nilay(nc-1)
     $              , ain(nc),   vin(nc),   vsn(nc)
     $                 , ein(:,nc),   Tsfn(nc)
     &        ,   hin(nc), hsn(nc)
     $                 ,        hin_max(nc-1),   nilay(nc) )
           endif
          enddo

         ! make sure thickness of cat 1 is at least hin_max(0)
         if ((hin(1).lt.hin_max(0)).and.(hin_max(0).gt.c0)) then
           ain(1) = vin(1)/hin_max(0)
           hin(1) = hin_max(0)
         endif

      endif
          
      end subroutine rebin_ice

c=======================================================================

      subroutine movedn(   Tf0
     $                  ,  ai1,   vi1,  vs1, ei1, Tsf1
     $                  ,  hi1,   hs1
     &                  , h1min,  nn1
     $                  ,  ai2,   vi2,  vs2, ei2, Tsf2
     $                  ,  hi2,   hs2
     &                  , h2min,  nn2       )

!---!-------------------------------------------------------------------
!---! Moves ice from cat 2 to cat 1.
!---! It is possible that the final thickness will be below the
!---! boundary of cat 1 if the ice has melted a lot, in which case
!---! it will be moved down again by a subsequent call
!---!
!---! author C. M. Bitz
!---!-------------------------------------------------------------------

      real (kind=dbl_kind), intent(in) ::
     &   Tf0           ! freezing temperature of ocean
     &,  h1min, h2min    ! minimum thickness range of cat 1, 2     (m)
      integer (kind=int_kind), intent(in) :: 
     &   nn1,   nn2    ! number of layers in cat 1, 2

      real (kind=dbl_kind), intent(inout) ::
     &   ai1, ai2         ! fraction of ice
     &,  vi1, vi2         ! volume per unit area of ice             (m)
     &,  vs1, vs2         ! volume per unit area of snow            (m)
     &,  hi1, hi2         ! ice thickness                           (m)
     &,  hs1, hs2         ! snow thickness                          (m)
     &,  Tsf1, Tsf2         ! ice/snow top surf. temperature          (K)

      real (kind=dbl_kind), intent(inout), dimension (:) ::
     &   ei1        ! energy of melting of ice per layer (J/m**2)
     &,  ei2       ! energy of melting of ice per layer (J/m**2)

      integer (kind=int_kind) :: M,layer,index,isub

      real (kind=dbl_kind) ::
     &   x(nn2)       ! for moving heat from 1 to 2        (J/m**2)
     &,  atemp              ! sum of ice frac.
      
      atemp = ai1+ai2
      if (atemp.le.0.0) return

      vi1 = vi1+vi2
      vs1 = vs1+vs2
      Tsf1 = Tsf1*ai1+Tsf2*ai2

      ai1 = atemp

      hs1 = vs1/ai1
      hi1 = vi1/ai1
      Tsf1 = Tsf1/ai1

      ! combine the nn2 layers of heat in cat 2 into nn1 portions
      M = nn2/nn1 
      do layer = 1,nn1
         x(layer) = c0
         do isub = 1,M
            index = (layer-1)*M+isub
            x(layer) = x(layer)+ei2(index)
         enddo
      enddo
         
      ! sum the heat
      do layer = 1,nn1
         ei1(layer) = ei1(layer)+x(layer)
      enddo
      
      call zerocat( Tf0, ai2, vi2,  vs2, ei2
     $            , Tsf2,hs2, hi2,h2min,  nn2 )

      end subroutine movedn

c=======================================================================

      subroutine moveup(     Tf0
     $                    ,  ai1,   vi1,  vs1, ei1, Tsf1
     $                    ,  hi1,  hs1
     &                    , h1min,  nn1
     $                    ,  ai2,   vi2,  vs2, ei2, Tsf2
     $                    ,  hi2,  hs2
     &                    , h2min,  nn2       )

!---!-------------------------------------------------------------------
!---! Moves ice from cat 1 to 2.
!---! It is possible that the final thickness will be above the 
!---! boundary of cat 2. If the ice needs to move up again, 
!---! it will do so in a subsequent call
!---!
!---! author C. M. Bitz
!---!-------------------------------------------------------------------

      real (kind=dbl_kind), intent(in) ::
     &   Tf0         ! temperature of ocean
     &,  h1min, h2min  ! minimum thickness range of cat 1, 2  (m)
      integer (kind=int_kind), intent(in) :: 
     &   nn1, nn2    ! number of layers in cat 1, 2

      real (kind=dbl_kind), intent(inout) ::
     &   ai1, ai2    ! fraction of ice
     &,  vi1, vi2    ! volume per unit area of ice             (m)
     &,  vs1, vs2    ! volume per unit area of snow            (m)
     &,  hi1, hi2    ! ice thickness                           (m)
     &,  hs1, hs2    ! snow thickness                          (m)
     &,  Tsf1, Tsf2    ! ice/snow top surf. temperature          (K)
     &,  ei1(nn1)     ! energy of melting of ice per layer (J/m**2)
     &,  ei2(nn2)     ! energy of melting of ice per layer (J/m**2)

      integer (kind=int_kind) :: M,layer,index,isub
      real (kind=dbl_kind) ::
     &   x(nn2)
     &,  atemp

      atemp = ai1+ai2
      if (atemp.le.0.0) return

      vi2 = vi1+vi2
      vs2 = vs1+vs2
      Tsf2 = Tsf1*ai1+Tsf2*ai2

      ai2 = atemp

      hs2 = vs2/ai2
      hi2 = vi2/ai2
      Tsf2 = Tsf2/ai2
         

      ! move energy of nn1 layers from 1 into the nn2 of 2
      M = nn2/nn1 
      do index = 1,nn1
         do isub  = 1,M
            layer = M*(index-1)+isub
            x(layer) = ei1(index)/M
         enddo
      enddo
      
      ! sum the heat
      do layer = 1,nn2
         ei2(layer) = ei2(layer)+x(layer)
      enddo
      
      call zerocat( Tf0, ai1, vi1,  vs1, ei1
     $            , Tsf1, hs1, hi1, h1min,  nn1 )

      end subroutine moveup

c=======================================================================

      subroutine zerocat( Tf0, ai, vi,  vs, ei
     $                   , Tsf
     & , hs, hi
     & , hmin,  nn )

!---!-------------------------------------------------------------------
!---! reset category variables in the case of no ice
!---!
!---! author C. M. Bitz
!---!-------------------------------------------------------------------

      real (kind=dbl_kind), intent(in) ::  Tf0      ! freezing temperature
      integer (kind=int_kind), intent(in) :: nn   ! number of layers in cat
      real (kind=dbl_kind), intent(in) :: hmin

      real (kind=dbl_kind), intent(inout) ::
     &   ai           ! fraction of ice
     &,  vi           ! volume per unit area of ice             (m)
     &,  vs           ! volume per unit area of snow            (m)
     &,  hi           ! ice thickness                           (m)
     &,  hs           ! snow thickness                          (m)
     &,  Tsf           ! ice/snow top surf. temperature          (K)
     &,  ei (nn)      ! energy of melting of ice per layer (J/m**2)

      integer (kind=int_kind) :: layer

      vi = c0
      vs = c0
      ai = c0
      hs = c0
      hi = hmin
      do layer = 1,nn
         ei(layer) = c0
      enddo
      Tsf = Tf0

      end subroutine zerocat

c=======================================================================

      end module ice_itd

c=======================================================================
