c $Id$
c=======================================================================
!---! Ice thickness distribution model with multi-layer thermodynamics.
!---! Routines to compute mechanical redistribution and strength.
!---!
!---! author C. M. Bitz
!---!
!---! See Bitz, C.M., and W.H. Lipscomb, 1999: 
!---! An energy-conserving thermodynamic model of sea ice,
!---! J. Geophys. Res., 104, 15,669-15,677. 
!---!     
!---! See Bitz, C.M., M.M. Holland, A.J. Weaver, M. Eby:
!---! Simulating the ice-thickness distribution in a climate model
!---! accepted to J. Geophys. Res.
!---! (prepublications copy available upon request)
!---!
!---! The author grants permission to the public to copy and use this
!---! software without charge, provided that this Notice and any statement
!---! of authorship are reproduced on all copies and any publications that
!---! result from the use of this software must (1) refer to the publications 
!---! listed above and (2) acknowledge the origin and author of the model.
!---! This software is without warranty, expressed or implied, and the
!---! author assumes no liability or responsibility for its use. 
c=======================================================================

      module ice_mechred

      use ice_model_size
      use ice_constants
      use ice_state
      use ice_itd

      implicit none

      logical (kind=log_kind) ::
     &   snow_into_ocn  ! if true, throw snow into ocean

      real (kind=dbl_kind), dimension (ilo:ihi,jlo:jhi) ::
     &   closing  ! lead closing rate         (1/s)
     &,  shear    ! strain rate II component  (1/s)
     &,  divu     ! strain rate I component, velocity divergence  (1/s)

      ! default redistributers, when thickness is at category mid-point
      real (kind=dbl_kind), dimension (ncat,ncat) ::
     &   Gammdf   !     area frac from cat i that goes into j
     &,  HGammdf  !   volume frac from cat i that goes into j
     &,  H2Gammdf ! H*volume frac from cat i that goes into j

      integer (kind=int_kind) ::
     &   ncrel    (ncat,ncat) ! helps transfer heat when ridging
     &,  kstrength            ! 1 for Rothrock 1975 strength formulation
     &,  kcolumn              ! 1 for 1  model

      real (kind=dbl_kind), parameter ::
     &   gstar = 0.15_dbl_kind ! if open water area > GSTAR then no 
                               ! mechanical redist.
     &,  Zfric = 17.0_dbl_kind ! frictional energy mult. factor for 
                               ! Roth. 75 strength 
     &,  Pstar = 2.75e4_dbl_kind ! empirical formula if not using 
                                 ! Roth. 75 strength  (N/m**2)
     &,  cK = 100.0_dbl_kind  ! max ridged ice thickness is 2*sqrt(cK*Hi) 
                              ! where Hi is orig thickness (m)
     &,  cpe = 450._dbl_kind ! if using Rothrock75 energetics argument for 
                    ! strength, coef. for potential energy term    (N/m**3)

      ! for water and heat budgets for 1  model
      real (kind=dbl_kind) ::
     &   mass2col  ! total ice/snow water mass entering 1  (kg/m2)
     &,  enth2col  ! total ice/snow enthalpy entering 1  (J)

c=======================================================================

      contains

c=======================================================================

      subroutine init_mechred

!---!-------------------------------------------------------------------
!---! Initialize constants for ridging
!---!-------------------------------------------------------------------

      integer (kind=int_kind) :: nc,k,l,m,layer, i,j
      logical (kind=log_kind) :: rowflg(ncat) ! flag for ridg. matrix row
      real (kind=dbl_kind) ::
     &   hin       (ncat)     ! ice thickness                        (m)
     &,  Hmean    (ncat)     ! a dummy variable at setup            (m)

      !-----------------------------------------------------------------
      ! matrix used to assist in heat transf. from cat i to j 
      !-----------------------------------------------------------------
      do l=1,ncat                     ! if nilay = { 2,4,8 }
         do m=1,l                     !     
            ncrel(l,m)=1              !    ncrel = | 1 2 4 |
         enddo                        !            | 1 1 2 |
         do m=l,ncat                  !            | 1 1 1 |
            ncrel(l,m)=nilay(m)/nilay(l)
         enddo
      enddo

      !-----------------------------------------------------------------
      ! vectors identifying first and last layer in each bin
      !-----------------------------------------------------------------
      layer1(1)=1                     ! if nilay = { 2,4,8 }
      layern(1)=nilay(1)              !   layer1 = { 1,3,7 } 
      do layer=2,ncat                 !   layern = { 2,6,16} 
         layer1(layer)=layern(layer-1)+1
         layern(layer)=layern(layer-1)+nilay(layer)
      enddo

      !-----------------------------------------------------------------
      ! default ridg matrices,  comp. all rows (rowflg=true)
      !-----------------------------------------------------------------
      ! assume ice thickness is mean of range for nc < ncat
      ! and 1m thicker than lower limit for ncat
      !-----------------------------------------------------------------
      do nc = 1,ncat
         do k = 1,ncat
            Gammdf(nc,k)   = c0
            HGammdf(nc,k)  = c0
            H2Gammdf(nc,k) = c0
         enddo
         rowflg(nc) = .true.  
         if (nc.lt.ncat) then
            hin(nc) = p5*(hin_max(nc-1) + hin_max(nc))
         else
            hin(ncat) = hin_max(ncat-1) + c1
         endif
      enddo
         
      call comp_matrices (    rowflg ,hin      ,Hmean
     $,                Gammdf ,HGammdf ,H2Gammdf )

      !-----------------------------------------------------------------
      ! initialize deformation and strength arrays
      !-----------------------------------------------------------------
      do j=jlo,jhi
      do i=ilo,ihi
        strength(i,j)=c0
        closing(i,j)=c0
        divu(i,j)=c0
        shear(i,j) = c0
      enddo
      enddo

      end subroutine init_mechred

c=======================================================================

      subroutine mechanical_redistr

!---!-------------------------------------------------------------------
!---! This routine works with ncat=1
!---! in fact, it will provide a source of open water from shear 
!---! deformation. See Stern et al, 1995 for information about 
!---! how and why this is done
!---!
!---! Keeps track of heat and fresh water flx to ocn
!---! if snow on ice that ridges is thrown into the ocean
!---!-------------------------------------------------------------------

      use ice_domain 
      use ice_flux
      use ice_timers
      use ice_grid
      use ice_calendar

      integer (kind=int_kind) :: i, j, nc, layer, l

      real (kind=dbl_kind) ::
     &   Gamm    (ncat,ncat) ! area frac from cat i that goes into j
     &,  H2Gamm  (ncat,ncat) ! H*volume frac from cat i that goes into j
         !  if using empirical formula for computing strength
     &,  ailcl               ! local sum of ice fraction over categories
     &,  vilcl               ! local sum of ice volume over categories (m)

          call ice_timer_start(6)  ! ridging 

      do j=jlo,jhi
        do i=ilo,ihi
          if (tmask(i,j)) then

            if (kcolumn.ne.1) call opening_closing(i,j)

            call to_column(i,j)

            call ridge( dt, closing(i,j), divu(i,j), Tf(i,j), 
     $               Fhnet(i,j), Fresh(i,j), Gamm, H2Gamm)
            
            if (kstrength.eq.1) then
              strength(i,j) = roth_strength( Gamm, H2Gamm )
            else
              ailcl = c0
              vilcl = c0
              do nc = 1,ncat
                ailcl = ailcl+ain(nc)
                vilcl = vilcl+vin(nc)
              enddo
              ! Hibler79, Eq 17, P=Pstar*h*exp(-C*(1-A)), h=effective thick 
              strength(i,j) = Pstar*vilcl*exp(-c20*(1-ailcl)) 
            endif
            
            call from_column(i,j)
            
          endif
        enddo
      enddo

          call ice_timer_stop(6)   ! ridging 

      end subroutine mechanical_redistr

c=======================================================================

      subroutine ridge( dt1, closng, epsi, Tf1, 
     $                  Fhnet1, Fresh1, Gamm, H2Gamm)

!---!-------------------------------------------------------------------
!---!    ridge ice due to convergence or shear deformation
!---!    note the timestep is dt1 for ridging among the cats 
!---!    however will subcycle with timestep dtsub if a cat 
!---!    would run out of ice if the timestep were > dtsub
!---!
!---!    throw snow on ice that ridges into ocean if snow_into_ocn=.true.
!---!-------------------------------------------------------------------

      real (kind=dbl_kind), intent(in) ::
     &   dt1
     &,  epsi       ! divergence of ice velocity            (1/s)
     &,  closng    ! closing from Flato & Hibler Eq. 9     (1/s)
     &,  Tf1       ! ocean freezing temperature

      real (kind=dbl_kind), intent(out) ::
     &   Fhnet1                ! heat given to ocean from snow         (N/m)
     &,  Fresh1                ! water given to ocean from snow          (m)
     &,  H2Gamm  (ncat,ncat) ! H*volume frac from cat i that goes into j

      real (kind=dbl_kind), intent(inout) ::
     &   Gamm    (ncat,ncat) ! area frac from cat i that goes into j

      integer (kind=int_kind) :: nc,k,layer,layk

      real (kind=dbl_kind) ::
     &   asum                ! total fraction of ice + open water
     &,  dtsub               ! subcycling timestep                     (s)
     &,  dtsubn              ! used to find subcycling timestep        (s)
     &,  dtaccum             ! accumulated time in this routine        (s)
     &,  dvisum              ! sum of dvi                              (m)
     &,  dvssum              ! sum of dvs                              (m)
     &,  deisum              ! sum of dei                              (J)
     &,  clsdt               ! closng * timestep
     &,  divudt              ! divergence * timestep

      real (kind=dbl_kind), dimension (0:ncat) ::
     &   W                   ! W factor, geometric comp. of ridging mode 
     &,  Wa                  ! part of W from participation func.
     &,  Psi                 ! rate of chng of area frac from ridg   (1/s)

      real (kind=dbl_kind), dimension (ncat,ncat) ::
     &   HGamm               ! volume frac from cat i that goes into j

      real (kind=dbl_kind), dimension (ncat) ::
     &   Hmean               ! mean thickness of ridged ice from cat nc
     &,  hin                  ! ice thickness                           (m)
     &,  hsn                  ! snow thickness                          (m)
     &,  dvi                 ! geometric factor for ice volume         (m)
     &,  dvs                 ! geometric factor for snow volume        (m)
     &,  dtf                 ! geometric factor, temp * frac area      (K)
     &,  ai_Tsfn              ! temperature * fractional area           (K)

      real (kind=dbl_kind), dimension (nmax,ncat) ::
     &   qin                  ! energy of melt ice per unit vol.   (J/m**3)
     &,  dei                 ! geometric factor for energy of melt     (J)

      dtsub   = dt1  ! begin assuming the subcyling step is the full timestep
      dtaccum = 0 

      if (kcolumn.eq.1) then ! add the g(h) div u term to ITD
        divudt=epsi*dt1
        if (divudt.gt.1.0) then
          print*, 'ice_mechred divudt>1!'
          stop
        endif
        ai0=ai0-divudt*ai0
        do nc = 1,ncat
          enth2col = enth2col+divudt*vsn(nc)*rLfs
          mass2col = mass2col-divudt*(rhoi*vin(nc)+rhos*vsn(nc))
          ain(nc) = ain(nc)-divudt*ain(nc)
          vin(nc) = vin(nc)-divudt*vin(nc)
          vsn(nc) = vsn(nc)-divudt*vsn(nc)
          do layer = 1,nilay(nc)
            enth2col = enth2col-divudt*ein(layer,nc)
            ein(layer,nc) = ein(layer,nc)-divudt*ein(layer,nc)
          enddo
        enddo
      endif

      asum = ai0
      do nc = 1,ncat
         asum = asum + ain(nc)
      enddo

      !-----------------------------------------------------------------
      ! increase open water and asum by opening rate
      !-----------------------------------------------------------------
      ai0=ai0+dt1*(epsi+closng)
      asum=asum+dt1*(epsi+closng)

      if (asum.le.c1) then  ! increase open water and quit
        ai0=ai0+c1-asum 
        if (kstrength.eq.1) 
     $      call ridge_matrices( hin, Hmean, Gamm, HGamm, H2Gamm ) 
        return
      endif

      ! if asum > 1 must ridge
 50   continue      

      !-----------------------------------------------------------------
      ! compute the ridge matrices and W factors, see docum.
      !-----------------------------------------------------------------
      call ridge_matrices( hin, Hmean, Gamm, HGamm, H2Gamm ) 
      call ridging_mode ( W, Wa, Gamm )

c      print*,'Wa ', (Wa(nc),nc=0,ncat)
c      print*,'W  ', (W(nc),nc=0,ncat)

      if (ncat.gt.1) then
      do nc = 1,ncat
         hsn(nc) = c0
         do layer = 1,nilay(nc)
            qin(layer,nc) = c0
         enddo
         if ((ain(nc).gt.puny) .and. (vin(nc).gt.0.))  then
            hsn(nc) = vsn(nc)/ain(nc)
            do layer = 1,nilay(nc)
               qin(layer,nc) = ein(layer,nc)*nilay(nc)/vin(nc)
            enddo
         endif
      enddo

      !-----------------------------------------------------------------
      ! compute terms in bracket of Theta (geom. term, see docum.)
      !-----------------------------------------------------------------
      dvisum = c0
      dvssum = c0
      do nc = 1,ncat-1
         dvi(nc) = -hin(nc)*Wa(nc)
         dvs(nc) = -hsn(nc)*Wa(nc)
         do k=1,nc
            dvi(nc) = dvi(nc) + Wa(k)*HGamm(k,nc)
            if (.not. snow_into_ocn)
     &      dvs(nc) = dvs(nc) + Wa(k)*Gamm(k,nc)*hsn(k)*Hmean(k)/hin(k)
         enddo
         dvisum = dvisum + dvi(nc)
         dvssum = dvssum + dvs(nc)
      enddo
      dvi(ncat) = -dvisum       ! guarantees ice volume conservation
      if (snow_into_ocn) then
        dvs(ncat) = -hsn(ncat)*Wa(ncat) ! only participation term 
      else
        dvs(ncat) = -dvssum       ! guarantees snow volume conservation
      endif

      !-----------------------------------------------------------------
      ! compute terms in bracket of Pi (geom. term, see docum.)
      !-----------------------------------------------------------------
      ! when ice ridges from cat k into cat nc heat is 
      ! transf. horiz. by layer w/o vert. mixing
      ! e.g., if cat k has 2 layers and cat nc has 4, heat from
      ! layer 1 of cat k is split evenly btwn layers 1 & 2 of cat nc
      !-----------------------------------------------------------------
      deisum = c0
      do nc = 1,ncat-1
         do layer = 1,nilay(nc)
            dei(layer,nc) = -Wa(nc)*hin(nc)*qin(layer,nc)/nilay(nc)
            do k = 1,nc
               layk = (layer+ncrel(k,nc)-1)/ncrel(k,nc)
               dei(layer,nc) = dei(layer,nc)+
     $              qin(layk,k)*Wa(k)*HGamm(k,nc)/nilay(nc)
            enddo
            deisum = deisum+dei(layer,nc)
         enddo
      enddo
      do layer = 1,nilay(ncat)
         dei(layer,ncat) = c0
         do k = 1,ncat-1
            layk = (layer+ncrel(k,ncat)-1)/ncrel(k,ncat)
            dei(layer,ncat) = dei(layer,ncat)+
     $           qin(layk,k)*Wa(k)*HGamm(k,ncat)/nilay(ncat)
         enddo
         deisum = deisum+dei(layer,ncat)
      enddo
c      write(*,*) 'Energy is conserved if deisum is zero: deisum= '
c     $     ,deisum

      !-----------------------------------------------------------------
      ! geometrical factor for treating ice/snow surface temp when ridging
      !-----------------------------------------------------------------
      do nc = 1,ncat
         dtf(nc) = - Tsfn(nc)*Wa(nc)
         do k=1,nc
           dtf(nc) = dtf(nc)+ Tsfn(k)*Wa(k)*Gamm(k,nc)
         enddo
      enddo

      endif ! ncat > 1

      !-----------------------------------------------------------------
      ! make sure that only ridge as much ice as is available, 
      ! otherwise reduce timestep
      !-----------------------------------------------------------------
      Psi(0) = closng*Wa(0)
      if ((Psi(0)*dtsub.gt.ai0) .and. (Psi(0).gt.0.)) then
         dtsubn = ai0/Psi(0)
         dtsub  = min(dtsub,dtsubn)
      endif
      do nc = 1,ncat
         Psi(nc) = closng*Wa(nc)
         if ((Psi(nc)*dtsub.gt.ain(nc)) .and. (Psi(nc).gt.0.)) then
            dtsubn = ain(nc)/Psi(nc)
            dtsub  = min(dtsub,dtsubn)
         endif
      enddo

      !-----------------------------------------------------------------
      ! finally update the prognostic variables
      !-----------------------------------------------------------------
      clsdt = closng*dtsub 

c      print*, 'clsdt',clsdt,closng,dtsub,dt1
      ai0   = ai0 + clsdt*W(0)
      asum  = ai0
      do nc = 1,ncat
         if (ncat.gt.1) then
         ai_Tsfn(nc) = ain(nc)*Tsfn(nc) + clsdt*dtf(nc)
         endif
         ain(nc)  = ain(nc) + clsdt*W(nc)
         asum = asum + ain(nc)
         if (ncat.gt.1) then
         vin(nc)  = vin(nc) + clsdt*dvi(nc)
         vsn(nc)  = vsn(nc) + clsdt*dvs(nc)
         if (snow_into_ocn) then
         ! Assuming snow falls into the ocn
         ! dvs is the volume of snow on ice that participates in ridging
         ! dvs <= 0.  
         ! add a positive quantity to Fresh1 (ocn gains water from snow)
         ! add a negative quantity to Fhnet1 (ocn looses heat to melt snow)
           Fresh1 = Fresh1 - closng*dvs(nc)*rhos !  kg/m^2 s
           Fhnet1 = Fhnet1 + closng*dvs(nc)*rLfs   
         endif
         do layer = 1,nilay(nc)
            ein(layer,nc) = ein(layer,nc) + clsdt*dei(layer,nc)
         enddo
         endif ! ncat > 1
      enddo

      if (ncat.gt.1) then
      do nc = 1,ncat
        if (ain(nc).gt.0.) then
          Tsfn(nc) = ai_Tsfn(nc)/ain(nc)
        else
          Tsfn(nc) = Tf1
        endif
      enddo
      endif ! ncat > 1

      !-----------------------------------------------------------------
      ! decide if time accumulated in ridge is sufficient to return
      !-----------------------------------------------------------------
      dtaccum = dtaccum + dtsub
      dtsub   = dt1 - dtaccum
      if (dtsub.gt.puny) go to 50

      end subroutine ridge

c=======================================================================

      subroutine ridging_mode( W, Wa, Gamm )

!---!-------------------------------------------------------------------
!---! compute W and Wa, factors for ridging mode 
!---! and participation function
!---!-------------------------------------------------------------------

      real (kind=dbl_kind), intent(in) ::
     &  Gamm    (ncat,ncat) ! area frac from cat i that goes into j

      real (kind=dbl_kind), intent(out) ::
     &   W       (0:ncat)    ! W factor, geometric comp. of ridging mode 
     &,  Wa      (0:ncat)    ! part of W from participation func.

      integer (kind=int_kind) ::  nc, k
      real (kind=dbl_kind) ::
     &   asum                ! sum of frac of ice + open water
     &,  Wsum                ! sum of W factors, for norm.
     &,  etar                ! norm. coef.
     &,  AUX(-1:ncat+1)      ! auxilary function

      !-----------------------------------------------------------------
      ! the Wa factor for cat nc is the area under the b(G) curve
      ! between G(nc-1) and G(nc) where G(nc) is the sum of ice 
      ! fractions upto and including cat nc, with G(-1)=0
      !-----------------------------------------------------------------

      AUX(-1)=c1

      AUX(0)=c0
      asum=ai0
      if (asum.lt.GSTAR)   AUX(0)=(1-asum/GSTAR)**2
      do nc=1,ncat
         AUX(nc)=c0
         asum=asum+ain(nc)
         if (asum.lt.GSTAR) AUX(nc)=(1-asum/GSTAR)**2
      enddo

      do nc=0,ncat
         Wa(nc)=AUX(nc-1)-AUX(nc)
      enddo

      !-----------------------------------------------------------------
      ! normalize W and Wa
      !-----------------------------------------------------------------
      Wsum = -Wa(0)
      do nc = 1,ncat
         W(nc) = -Wa(nc)
         do k = 1,nc
            W(nc) = W(nc) + Wa(k)*Gamm(k,nc)
         enddo
         Wsum = Wsum + W(nc)
      enddo

      etar = -c1/Wsum  ! normal. coef keep sign of Wa positive
      
      do nc=1,ncat
         W(nc)  = etar*W(nc)
         Wa(nc) = etar*Wa(nc)
      enddo
      Wa(0) = etar*Wa(0)

      W(0)  = -Wa(0)

      end subroutine ridging_mode

c=======================================================================

      subroutine ridge_matrices( hin, Hmean, Gamm, HGamm, H2Gamm ) 

!---!-------------------------------------------------------------------
!---! compute Gamm,HGamm,H2Gamm for ridge and strength for true thicknesses
!---!-------------------------------------------------------------------

      real (kind=dbl_kind), intent(out) ::
     &   hin      (ncat)      ! ice thickness                           (m)
     &,  Hmean   (ncat)      ! mean thickness of ridg ice from cat nc  (m)
     &,  Gamm    (ncat,ncat) ! area frac from cat i that goes into j
     &,  HGamm   (ncat,ncat) ! volume frac from cat i that goes into j
     &,  H2Gamm  (ncat,ncat) ! H*volume frac from cat i that goes into j
       ! HGamm is used in ridge and H2Gamm is used in roth_strength

      integer (kind=int_kind) :: k,nc
      logical (kind=log_kind) :: rowflg(ncat) ! true if cat has nonzero area

      do nc = 1,ncat
         if ((ain(nc).gt.puny) .and. (vin(nc).gt.0.))  then
            hin(nc) = vin(nc)/ain(nc)
            rowflg(nc) = .true.
            do k = 1,ncat
              Gamm(nc,k)   = c0
              HGamm(nc,k)  = c0
              H2Gamm(nc,k) = c0
            enddo
         else
            hin(nc) = max(hin_max(nc-1),hi_min)
            rowflg(nc) = .false.
            do k = 1,ncat
              Gamm(nc,k)   = Gammdf(nc,k)
              HGamm(nc,k)  = HGammdf(nc,k)
              H2Gamm(nc,k) = H2Gammdf(nc,k)
            enddo
         endif
      enddo
         
      call comp_matrices  ( rowflg, hin, Hmean,
     $                      Gamm, HGamm, H2Gamm )

      end subroutine ridge_matrices

c=======================================================================

      subroutine comp_matrices( rowflg, hin, Hmean,
     $                           Gamm, HGamm, H2Gamm )

!---!-------------------------------------------------------------------
!---! compute Gamm, HGamm, H2Gamm
!---! HGamm is used in ridge and H2Gamm is used in roth_strength
!---! Gamm(nc,k) distributes ice that participates in ridging
!---! from cat nc into a distrib. of cats k
!---! n1 is the smallest k with nonzero fract
!---! n2 is the largest  k with nonzero fract
!---! ice that partic. has thickness hin(nc)
!---! and it ridges up to linear distrib. between 
!---! 2*hin(nc) and 2*sqrt(cK*hin(nc))
!---!-------------------------------------------------------------------

      logical (kind=log_kind), intent(in) ::
     &   rowflg(ncat)     ! true if cat has nonzero fractional area
      real (kind=dbl_kind), intent(in) ::
     &   hin      (ncat)      ! ice thickness                           (m)

      real (kind=dbl_kind), intent(out) ::
     &   Hmean   (ncat)      ! mean thickness of ridg ice from cat nc  (m)
     &,  Gamm    (ncat,ncat) ! area frac from cat i that goes into j
     &,  HGamm   (ncat,ncat) ! volume frac from cat i that goes into j
     &,  H2Gamm  (ncat,ncat) ! H*volume frac from cat i that goes into j

      integer (kind=int_kind) :: 
     &   nc, k
     &,  n1, n2           ! lowest and highest cat to which ice ridges
      real (kind=dbl_kind) ::
     &   rcKH             ! sqrt(cK*hin)                          (m**.5)
     &,  bet              ! 2*(cK-hin)                                (m)

      do nc = 1,ncat
         rcKH      = sqrt(cK*hin(nc))
         Hmean(nc) = hin(nc)+rcKH
         bet       = c2*(cK-hin(nc))

         if (rowflg(nc)) then
            n1 = nc  ! first guess
            n2 = n1  ! first guess
            do k = nc+1,ncat
               if (c2*hin(nc).ge.hin_max(k-1)) n1 = k
            enddo
            if (n1.eq.ncat) then
               ! case when all ice ridges into thickest cat
               Gamm(nc,n1)   = hin(nc)/Hmean(nc)
               HGamm(nc,n1)  = hin(nc)
               H2Gamm(nc,n1) = hin(nc)*Hmean(nc)
            elseif (2*rcKH.le.hin_max(n1)) then
               ! case when all ice ridges into just one cat
               Gamm(nc,n1)   = hin(nc)/Hmean(nc)
               HGamm(nc,n1)  = hin(nc)
               H2Gamm(nc,n1) = hin(nc)*Hmean(nc)
            else
               ! general case
               n2 = ncat
               do k = ncat-1,n1+1,-1
                  if (c2*rcKH.lt.hin_max(k)) n2 = k
               enddo
               Gamm(nc,n1)   = (hin_max(n1)-c2*hin(nc))/bet
               HGamm(nc,n1)  = Gamm(nc,n1)*(hin(nc)+hin_max(n1)/c2)
               H2Gamm(nc,n1) = HGamm(nc,n1)*(hin(nc)+hin_max(n1)/c2)

               Gamm(nc,n2)   = (c2*rcKH-hin_max(n2-1))/bet
               HGamm(nc,n2)  = Gamm(nc,n2)*(hin_max(n2-1)/c2+rcKH)
               H2Gamm(nc,n2) = HGamm(nc,n2)*(hin_max(n2-1)/c2+rcKH)

               do k = n1+1,n2-1
                 Gamm(nc,k)   = (hin_max(k)-hin_max(k-1))/bet
                 HGamm(nc,k)  = Gamm(nc,k)*hin_mp(k) 
                 H2Gamm(nc,k) = HGamm(nc,k)*hin_mp(k) 
               enddo
            endif
         endif
      enddo

      end subroutine comp_matrices

c=======================================================================

      real function roth_strength( Gamm ,H2Gamm )

!---!-------------------------------------------------------------------
!---! Compute the ice strength based on Rothrock, 1975 and also see FH95
!---! does not make sense to do this unless the ice that participates in 
!---! ridging is well resolved -- must have about 5 categories, 10 would 
!---! be better
!---!-------------------------------------------------------------------

      real (kind=dbl_kind), intent(in) ::
     &   Gamm    (ncat,ncat) ! area frac from cat i that goes into j
     &,  H2Gamm  (ncat,ncat) ! H*volume frac from cat i that goes into j

      integer (kind=int_kind) :: k,nc
      real (kind=dbl_kind) ::
     &   W       (0:ncat)    ! W factor, geometric comp. of ridg mode 
     &,  Wa      (0:ncat)    ! part of W from participation func.
     &,  hin      (ncat)      ! ice thickness                           (m)
     &,  peterm              ! contribution from potential energy   (m**2)

      roth_strength = c0
      if (ai0.ge.gstar) return

      ! strictly speaking, should recompute Gamm and H2Gamm matrices here,
      ! but they would be only slightly different from what they are
      ! now and it does not affect conservation prop.

      call ridging_mode( W ,Wa ,Gamm )

      do nc = 1,ncat
         if ( (ain(nc).gt.puny) .and. (vin(nc).gt.0.) )  then
            hin(nc) = vin(nc)/ain(nc)
         else
            hin(nc) = max(hin_max(nc-1),hi_min)
         endif
      enddo

      peterm = c0
      do nc = 1,ncat
         peterm = peterm - Wa(nc)*hin(nc)*hin(nc)
         do k = 1,nc
            peterm = peterm + Wa(k)*H2Gamm(k,nc)
         enddo
      enddo
      peterm = max(c0,peterm)

      ! factor of Zfric accounts for frictional energy
      roth_strength = peterm*cpe*Zfric  

      end function roth_strength

c=======================================================================

      subroutine opening_closing(i,j)

!---!-------------------------------------------------------------------
!---! divergence/shear quantities for ridging
!---!-------------------------------------------------------------------

      use ice_kinds_mod            ! variable precision module
      use ice_grid
      use ice_calendar
      use ice_dyn_evp

      integer (kind=int_kind) :: i, j, nc
      real (kind=dbl_kind) :: 
     &    factor, UTE, UTW, VTN, VTS
     &,   e11, e22, e12, shear2
     &,   delt, delta_pl
     &,   asum

      !-----------------------------------------------------------------
      !         e11 = du/dx
      !         e22 = dv/dy
      !         e12 = 0.5*(du/dy+dv/dx)
      !  divergence = du/dx + dv/dy = e11 + e22 
      !  shear      = sqrt( (du/dx - dv/dy)**2 + (du/dy+dv/dx)**2 )
      !             = sqrt( ( e11  -  e22 )**2 + (   2.*e12  )**2 )
      !-----------------------------------------------------------------

        factor = p5*dxtr(i,j)*dytr(i,j)
        UTE =  factor*HTE(i,j)
        UTW = -factor*HTE(i-1,j)
        VTN =  factor*HTN(i,j)
        VTS = -factor*HTN(i,j-1)

      ! the strain rate tensor is computed without metric terms consistent
      ! evp dynamics and mpdata advection
        e11 = UTE*(u(i,j)+u(i,j-1)) + UTW*(u(i-1,j) + u(i-1,j-1))
        e22 = VTN*(v(i,j)+v(i-1,j)) + VTS*(v(i,j-1) + v(i-1,j-1))
        e12 = p5*(VTN*(u(i,j)+u(i-1,j)) + VTS*(u(i,j-1) + u(i-1,j-1))
     $           + UTE*(v(i,j)+v(i,j-1)) + UTW*(v(i-1,j) + v(i-1,j-1)))
          
        divu(i,j) = e11 + e22 

        shear2 = (e11-e22)**2 + (c2*e12)**2 
        shear(i,j) = sqrt( shear2 )
              
      ! Replaced ecci with p25, since ecci=1= 4/e^2 in ice_dyn_evp.F
      ! The value of ecci here is 1/e^2, or 0.25.  8 May 2001 JLS
        delt = divu(i,j)**2+p25*shear2  ! delta from Hibler, 1979
        delta_pl = sqrt(delt)
          
      ! unfortunately the second order accurate advection 
      ! schemes do not preserve asum=1-(e11+e22)*dt because they have higher 
      ! order corrections that I have not included in the strain 
      ! rates above. Would be best to figure out how to properly 
      ! compute the strain rates to be consistent with the advection
      ! scheme, but instead adjust epsi here to compensate:
      ! ignoring effect on delta_pl
          asum = aice0(i,j)
          do nc = 1,ncat
            asum = asum + aicen(i,j,nc)
          enddo
          
          divu(i,j) = ( c1 - asum )/dt

      !-----------------------------------------------------------------
      ! compute closing, aka alpha_r(theta) from Thorndike et al., 1975
      !-----------------------------------------------------------------

      ! following Stern et al., 1995     - MAX SHEAR DEFORMATION
c        closing(i,j)=p5*(delta_pl-divu(i,j)) 
c        closing(i,j)=max(closing(i,j),-divu(i,j))

      ! following FH95                   - INTERM. SHEAR DEFORMATION
c        closing(i,j)=p25*(delta_pl-abs(divu(i,j)))-min(divu(i,j),c0)
c        closing(i,j)=max(closing(i,j),-divu(i,j))

      ! Bill Hibler suggesting           - NO SHEAR DEFORMATION
      ! to contribute to ridge building
c        closing(i,j)=-min(divu(i,j),c0)


      ! Strike out on my own  - BETWEEN INTERM. and NO SHEAR DEFORMATION
           closing(i,j)=0.1125_dbl_kind*(delta_pl-abs(divu(i,j)))
     $         -min(divu(i,j),c0)
           closing(i,j)=max(closing(i,j),-divu(i,j))

      end subroutine opening_closing

c=======================================================================

      end module ice_mechred

c=======================================================================
